;; AGENTS Model V1.0.0 (build 140524) @NetLogo version 6.4.0

extensions [ gis csv bitmap]

breed [terraces terrace] ;; Represents agricultural areas for grapevines growing
breed [farms farm] ;; Represents farms. A farm owns vineyards. Each 'farm' is a distinct agent in the model.
breed [mounds mound] ; represent runoff patches areas
breed [runoffs runoff] ;<<Debud Mode>>


globals [
  is-setup-error ; flag for error during setup to allow print error to user UI
  elevation ; Stores the elevation raster calculated from a DEM ASCII file.
  slope    ; Stores the slope raster calculated from the elevation raster using a 3x3 kernel.
  aspect   ; Stores the aspect calculated from the slope raster using bilinear interpolation.
  city-border-coordinates  ; Holds coordinates of city border patches.
  catchment-border-coordinates  ; Holds coordinates of catchment area border patches.
  next-terrace-id  ; Global variable to assign unique IDs to each terrace.
  potential-terraces ; Contains all potential grapevine (grape crop-type) locations for terraces based on preset conditions.
  global-terrace-created ; Boolean flag to track if a terrace was created in the current tick.

  precipitation-data ; List to store the precipitation data from the CSV file
  evaporation-data ; List to store the evaporation data from the CSV file

  current-year-index ; Index of the year currently being read from the CSV file
  current-day-index ; Index of the day currently being read from the CSV file

  ; Following the reconstruction of precipitation data during setup, the variables Day and current-month are initialized to 1, representing the start of the list rather than January 1st;
  ; the actual current date is reflected by season-start-day and season-start-month as the model progresses
  current-month ; current month (january as 1, february as 2...december as 12).
  day ; day of the month (value between 1 and 31 depends on the current month)

  year ; tracking year of Byzantine Shivta from 535 AD to 565 AD spaning 30 years of average vine life-span
  days-in-month

  ;set globals for deifning the season start date  (used for the precipitation data reconstructing cutoff point)
  season-start-day ;
  season-start-month ;
  ;season-end-day ;
  ;season-end-month ;

  ; track time for UI (user interface) only
  current-UI-day
  current-UI-month
  current-UI-year

  total-precipitation ;total amount of precipitation recorded on the model setup period
  total-annual-catchment-runoff ; total amount of runoff  created over all 1 year rainfall events. Resets every year
  yearly-total-precipitation ; current year annual total rainfall
  total-evaporation ; Total amount of evaporation recorded over the model run
  yearly-total-evaporation ; Total evaporation for the current year
  high-rainfall-count ; calculated if flood conditions met. Treshhold set in code level as >8mm. Calculate daily)
  daily-rainfall-amount; amount of rain per each day (tick in model level)
  monthly-total-rainfall-amount ; Total amount of rainfall recorded for that month
  daily-evaporation-amount ; amount of surface water evaporation rate per each day (tick in model level)
  antecedent-rainfall ; Keeps track of whether it rained the previous day
  climate-adjustment-factor ;adjust daily-rainfall-amount according to user defined special climate conditions (wetter / same-as-present / drier climate conditions

  ;max-terrace-ponding ; set maximum height to water accumilating in a terrace according to terrace wall height (in mm)
  terrace-creation-period ; set number of days intervals that farms create terrace. Example: if 90, then one terrace is created per farm every 90 days (ticks mod terrace-creation-period = 0)
  daily-infiltration-amount ; global daily infiltration rate of terrace water ponding infiltrate to the terrace soil, accumulate as terrace avaiable water storage (terrace variable: terrace-AWS)
  runoff-initial-abstraction ; global value indicating minimum amount of precipitation required to start runoff

  months-passed ; Number of months that have passed since the model start (to indicate a one year of vines grow)
  current-season ; Stores the current season
  random-year-indices ; List to store random indices for selecting years from the CSV file acording to the number of years the user selected to run the model
  next-farm-id ;; will keep track of the ID to be assigned to the next new farm

  runoff-cf ;; runoff calibration factor. This is determined during model calibration and validation steps an is scenario specific; for the Zetan scenario, this was found to be 1.567
  ;num-of-terraces-per-season ;; set number of terraces each farm creates when season start
  ;;external procedures global variables
  last-used-shade

  ;; agriculture labor costs (in kg wheat)
  build-grapevine-terrace
  build-wheat-field
  prepare-harvest-runoff-field
  vineyard-planting
  vineyard-establishment-phase
  vineyard-season-labor
  wheat-season-labor
  ;grapevine-infrastructure-maintenance
  ;wheat-infrastructure-maintenance
  wheat-sow-amount
  maintenance-cost
  dunam-to-patch-ratio

  ; wine production related variables
  agrisys-annual-wine-production ;aggragation of all active grape terraces annual yield in liters of wine
  agrisys-annual-wine-production-list ; pair of model year and annual agri-system total wine production
  wine-to-grape-weight-ratio ;how many liters of wine produced from 1 kg of grapes

  ; decision-making related


]

patches-own [
  ; Patch attributes derived from GIS data using ArcGIS Desktop V10.8.2 andor  NetLogo calculations:
  patch-elevation  ; Elevation value from DEM data.
  patch-slope-deg     ; Slope in degrees, calculated from GIS data.
  patch-slope-pr      ; Slope in percentage rise, calculated from GIS data.
  patch-aspect     ; Aspect in degrees, calculated from GIS data.
  patch-face  ; Compass direction the patch is facing.
  is-catchment-area ; Boolean indicating if the patch is within the catchment area.
  is-catchment-border ; Boolean indicating if the patch is on the catchment border.
  is-city-area ; Boolean indicating if the patch is within the city area.
  is-city-border ; Boolean indicating if the patch is on the city border.
  stream-order ; Stream order in the research area, with lower values for tributaries.
  stream-id  ; Identifier for the stream based on the 'arcid' from the stream network ArcGIS shapefile.
  is-potential-wheat-field ; patches that can be set as wheat growing fieds by farms. Near streams of order 3 or 4, 1st,2nd or 3rd neighbour row to the stream patch
  is-agriculture-terrace ; Boolean indicating if the patch is an agricultural terrace.
  patch-terrace-id ; Unique identifier for a terrace.
  patch-soil-index  ; Soil type of the patch.
  patch-runoff-coefficient  ; Runoff coefficient of the patch.
  patch-flow-direction ; Flow direction based on steepest downslope neighbor.
  patch-flow-accumulation ; Flow accumulation indicating water accumulation.
  patch-sub-basin-id ; Identifier for the sub-basin the patch belongs to.
  patch-type ; Can be "empty", "agriculture", or "urban".
  coverage-status ; Can be "natural", "urban", "vineyard or "clear with mounds".
  patch-runoff-cluster-terrace-id ; Tracks the associated terrace ID for runoff patches.
  is-tributary ; Boolean indicating if the patch is part of a tributary.
  is-main-river ; Boolean indicating if the patch is part of a main river.
  runoff-threshold  ; New patch attribute for runoff threshold, initialized to 0.0
  CN1S ; curve number index value according to soil group value of for dry soil moisture state
  CN2 ; curve number index value according to soil group value of for wet soil moisture state
  CN3 ; curve number index value according to soil group value of for saturated soil moisture state
  CN2S ; curve number index value according to soil group value of for wet soil moisture state with slope adjustments
  SAMC1 ; potential maximum retention for soil moisture state dry
  SAMC2 ; potential maximum retention for soil moisture state wet
  IaAMC1 ; initial abstraction for soil moisture state dry
  IaAMC2 ; initial abstraction for soil moisture state wet
  ;patch-effective-CN ;curve number altered value based on patch slope-pr and patch moisture state values
  patch-AMC ; AMC - antecedent moisture condition; values: 1 = 'dry' and  2 = 'wet'
  patch-rank ;ranking system for best wheat field location as a function of yield (values range 0 to 11; with 0 the least favourable location by farm's needs)
  patch-runoff-storage ; debug mode variable
  patch-moisture-level ; This list will track the daily water balance on each patch.



]

terraces-own [
  terrace-id ; Same as the terrace ID of the patch where the vineyard is located.
  terrace-sub-basin-id ; Same as the sub-basin ID of the patch where the vineyard is located.
  owned-by-farm ; Stores the farm-id of the farm that created this terrace.
  part-of-vineyard-number  ; New attribute to identify which vineyard the terrace is part of
  terrace-ponding-water  ;; Water accumulated in a terrace from rain or flowing runoff
  max-ponding        ;; Height of the terrace, defaults to 500, changeable only in code
  crop-type ; "none", "grapevine", or "wheat"
  terrace-AWS ; terrace avaiable water storage
  annual-terrace-AWS-list  ; List to recorded terrace available water (underground storage) on annual basis
  annual-terrace-yield-list ; List to recorded terrace yield on annual basis
  annual-terrace-yield-efficiency ; List to recorded terrace yield efficiency on annual basis; Efficiency = (annual yieald) / (max potential yield of crop-type)
  total-terrace-yield ; sum of all years (model run period) crop yield (kg)
  average-terrace-yield ;average of all years (model run period) crop yield (kg)
  is-active ; declare if a terrace (grapevine or wheat) are in use; if FALSE then no yield is calculated for these terrace against Farm total and no labor costs applied

]


farms-own [
  farm-id ; a unique identifier for each farm
  owned-terraces ; a list of terrace-ids owned by the farm
  owned-vineyards ; a list of vineyard numbers associated with the farm
  total-grape-yield-kg ; total yield accumulated over model run years from all grapevine terraces owned by the farm
  total-wheat-yield-kg ; total yield accumulated over model run years from all wheat terraces owned by the farm
  total-grape-terraces ; total number of terraces a farm owns that are for growing grapevines
  total-wheat-terraces ; total number of terraces a farm owns that are for growing wheat
  wheat-storage ; varible that stores total wheat yield of all farm wheat fields accumulated annually. This is also from where the farm deduct wheat to "pay" for agriculture activities
  grapes-storage ; varible that stores total grapevines yield of all farm wheat fields accumulated annually
  farm-potential-grapevine-terraces ; variable that stores all potential terraces for that farm with relationeship to its distance from the farm, incline and patch-rank value
  farm-potential-wheat-fields ; variable that stores all potential fields for that farm with relationeship to its distance from the farm, incline and patch-rank value
  current-season-wheat-yield-kg ; Wheat yield for the current season
  wheat-failure-records ; Record of farm's success/failure of average wheat yield
  grape-failure-records ; Record of farm's success/failure of average grapevines yield
  annual-avg-wheat-yield ; Record of average wheat yield over all active wheat fields at growing season end
  annual-avg-grape-yield ; Record of average grapevine yield over all active grapevine terrace at growing season end

]

runoffs-own [
  runoff-amount  ;; Amount of water each runoff agent carries, represents the flowing runoff
]


; Comprehensive setup procedure that initializes the model environment and data.
to load-arcgis-data
  print date-and-time
  print "Starting FULL setup process. that will take 3-4 minutes so please be patient"
  clear-all
  setup-world ; Configures map size, patch size, and world topology.
  init-globals ; Initializes global variables.
  init-variables ; Sets initial values for patch and breed variables.
  print date-and-time
  print "Finished world setup and init variables"

  ; Processes and assigns GIS data to patches, and exports data to a text file.
  print "Start loading data from vector ArcgIS files to patch level attributes"
  assign-shapefiles-data-to-patches ; set patches attributes based on data (files) loaded and analyzed
  print date-and-time
  print "Finished loading data from vector ArcgIS files to patch level attributes"

  print "Start exporting patches data to external csv file (this is later used to upload data to patches from the Quick-Setup process)"
  export-static-world-patches-values  ; Export patch data to a text file
  print date-and-time
  print "Finished exporting patches data to external csv file (this is later used to upload data to patches from the Quick-Setup process)"


  wait 1
  print ""
  print "Model ready for user to run SETUP"

  reset-ticks
end

; Simplified setup using pre-processed patch data.
to setup
  print date-and-time
  print "Starting quick setup process"
  print "Starting world setup and init variables"
  clear-all
  setup-world ;setup map size, pixel size and wrap mode
  init-globals ; initialize global variables
  init-variables ; setup patch or breeds variables first values to avoid errors during model run
  print date-and-time
  print "Finished world setup and init variables"

  print "Starting load external CSV file"
  load-patch-values ; Loads patch data from a pre-processed external file.
  print date-and-time
  print "Finished load external CSV file"

  print "Starting setting streams and city and locating potential terracess area"
  set-urban-area ;; set city area as non-agriculture area
  classify-streams ; Classify streams network by stream-order values to tributaries and main rivers
  set potential-terraces find-all-potential-terraces

  setup-potential-wheat-fields ; set potential wheat fields
  print date-and-time
  print "Finished setting streams and city and locating potential terracess area"

  print "Starting loading climate CSV files data to model"
  ; load percipitation data and initialize related variables
  load-precipitation-data ; Load precipitation data from the CSV file
  reconstruct-precipitation-data ;build one season of precipitation of two consecutive years according to season-start-month and season-start-day (season start date)
  set-climate-adjustment-factor
  load-evaporation-data ; Load evaporation data from the CSV file
  reconstruct-evaporation-data ;build one season of evaporation of two consecutive years according to season-start-month and season-start-day (season start date)

  if consecutive-droughts-period > model-run-period [
    set is-setup-error true ;flag error during setup mode

    ; resetting sliders so setup can continue properly
    ifelse model-run-period > 1 [
      set consecutive-droughts-period model-run-period - 1
    ] [set consecutive-droughts-period 1]
  ]
  initialize-time-indexes ; Initialize the indexes used for tracking time in the model
  print date-and-time
  print "Finished loading climate CSV files data to model"

  print "Start coloring map and visualize map features"
  color-topographic-map
  display-geographic-features
  print date-and-time
  print "Finished coloring map and visualize map features"

  print "Start constructing farms"
  construct-farms ;; create numbers of farms in the model as inidicated by user-controlled interface slider

  print date-and-time
  print "Finished constructing farms"

  print "Start setting boarder patches as non catchment area, updating current season and setting runoff threshold for all patches"
  ; Set is-catchment-area to FALSE for patches at the edge of the world. This serves when runoff flows on the terrain as a result of rainfall event
  ask patches with [pxcor = min-pxcor or pxcor = max-pxcor or pycor = min-pycor or pycor = max-pycor] [
    set is-catchment-area FALSE
  ]

  update-current-season ; set correct season
  set-patches-curve-number-index ;set patches all CN values (CN1,CN2,CN3) slope value adjusted for all moisture conditions
  setup-patch-ranks
  ;setup-farm-potential-grapevine-terraces

  ; set farms agriculture status starting-point
  repeat 20 [check-and-create-wheat-terraces] ; create 20 unint of wheat fileds per farm (~18 dunams)
  evaluate-owned-wheat-terraces ; pay for the previously created farm's wheat fields
  expand-vineyards ;with left wheat sotrage farms maximize grape terraces it builds
  update-owned-terraces ; update farm's terraces list with all newely founded grapevine terraeces and wheat fields
  set-mounds-terrain-scheme ; set runoff harvest area terrain (visual representation)
  update-farms-icons ; update farm color to visualize its economic status


  ;ask patches with [is-potential-wheat-field and not any? terraces-here] [set is-potential-wheat-field false]
  clear-all-plots

  let legend bitmap:import "Legend.png" ; load map legend image
  bitmap:copy-to-drawing legend 1 505 ; set legend location, at map bottom left corner

  print date-and-time
  print "Finished setting boarder patches as non catchment area, updating current season and setting runoff threshold for all patches"

  wait 1
  print ""
  print "Model ready for user"
  if is-setup-error [
    print ("Error occur during setup!!!")
    print ("Model run period slider value must be greater or equal then consecutive-droughts slider value!")
    print (word "Re-setting consecutive-droughts value to a proper value of " consecutive-droughts-period)
  ]

  reset-ticks
end


; Main execution loop of the model
to go
  if ticks < 1 [
    ;setup-farm-potential-grapevine-terraces
    ; Use 'map' to add 1967 to each year in the list
    let adjusted-years map [i -> i + 1952] random-year-indices
    ; print climat years order to UI (this is for debugging purpose only)
    print (word "random climate years order list: " adjusted-years)
    print (word "Started " model-run-period " years run: " date-and-time)
    print (word "climate factor: " climate-adjustment-factor)
    ;print-active-grapevine-terraces-info
    reset-timer
  ]

  ; call daily (every tick) procesures
  update-owned-terraces
  update-precipitation
  call-runoff-flow
  update-time
  update-UI-tracked-time
  update-current-season ; set correct season
  set-terraces-icons

   ; call annual (end of season) procesures
  if day = 31 and current-month = 12 [
    ask terraces [
      set annual-terrace-AWS-list lput round(terrace-AWS) annual-terrace-AWS-list
      set terrace-AWS 0
    ]
    terrace-yield ; initiate terrace yield calculation for end of season (year) based on terrace AWS and crop-type
    ;print (word "finished  terrace-yield at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    update-annual-terrace-yield-efficiency
    ;print (word "finished  update-annual-terrace-yield-efficiency at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    update-terrace-total-yields
    ;print (word "finished  update-terrace-total-yields at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    calculate-farm-yields
    ;print (word "finished  update-terrace-total-yields at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    calculate-total-wine-production
    ;print (word "finished  calculate-total-wine-production at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    update-farm-annual-average-yields
    ;print (word "finished  update-farm-annual-average-yields at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""
    update-farms-failure-records
    ;print (word "finished  update-farms-failure-records at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""

    report-end-of-run-grape-yield ; Plot grape total yield per farm
    report-end-of-run-wheat-yield ; Plot wheat total yield per farm
    ;print (word "finished  report-end-of-run-grape-yield and  report-end-of-run-wheat-yield at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""

  ]

  ;; Reset total runoff monitor at calender year start.
  if current-month = 12 and day = 31  [
   set total-annual-catchment-runoff 0
  ]



    ; call annual (start of season) procesures (except first and last year)
  if  ticks > 1 and day = 1 and current-month = 1 and ticks != ((model-run-period * 365) - 1)[
    evaluate-owned-wheat-terraces ;pay from farm's wheat storage for coming season owned wheat fields labor costs. if not enough wheat, those unpayed-for wheat fields are set not-active and coming season yield is 0.
    ;print (word "finished  evaluate-owned-wheat-terraces at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""


    evaluate-owned-grapevine-terraces ;pay from farm's wheat storage for coming season owned grapevine terraces labor costs. if not enough wheat, those unpayed-for grapevine terraces are set not-active and coming season yield is 0.
    ;print (word "finished  evaluate-owned-grapevine-terraces at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""

    update-farms-icons ;; set the economical status of the farms based on their ability to pay-for owned agriculture area and expand thier agriculture area
    ;print (word "finished  update-farms-icons at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""

    farm-agriculture-expansion ;; farms decideds based on previous seasn success (or failure) in grape an wheat yields of their agriculture area if to expand or not and which crop to expand if so
    ;print (word "finished  farm-agriculture-expansion at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""


    set-mounds-terrain-scheme
    ;print (word "finished  set-mounds-terrain-scheme at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
    ;print ""


  ]


  if ticks = ((model-run-period * 365) - 1) [
;  if ticks = ((model-run-period * 365)) [
    update-wine-production-plot
    ;print-active-grapevine-terraces-info
    print (word "Finished " model-run-period " years run: " date-and-time)
    print ""
    calculate-run-time
    ;if number-of-farms > 0 [print-farm-total-yields]
    stop
  ]

  tick

end


; Configures the initial world settings.
to setup-world
  resize-world 0 264 0 151  ; Matches the world size to the dimensions of the DEM data.
  set-patch-size 4.4
  __change-topology false false ; Sets the world to be unbounded (no wrapping).
end

; Initializes global variables to their default states.
to init-globals

  set is-setup-error false


  set day 1
  set current-month 1
  set year 535 ; Set start year to peak of Shivta glory (mid 6th c. AD)

  set season-start-day 1
  set season-start-month 8
  ;set season-end-day 31
  ;set season-end-month 7

  set current-UI-day season-start-day
  set current-UI-month season-start-month
  set current-UI-year year


  ;set max-slope-for-terrace 1
  ;set runoff-ratio 20
  set next-terrace-id 1
  set global-terrace-created false ;

  set high-rainfall-count 0 ;set floods count to 0
  set daily-rainfall-amount 0 ;
  set daily-evaporation-amount 0;
  set antecedent-rainfall false ;set previous day did not rained status
  set total-precipitation 0 ; Initialize total-precipitation to 0
  set total-annual-catchment-runoff 0; Initializing runoff counter to 0
  set yearly-total-precipitation 0 ; Initialize this (current) year total-precipitation to 0
  set total-evaporation 0
  set yearly-total-evaporation 0
  set monthly-total-rainfall-amount 0; reseting current month total amount of rainfall to 0 at model start
  set months-passed 0 ; Initialize months-passed to 0
  set current-season "" ; Initialize current season as empty name
  set random-year-indices []
  ;set max-terrace-ponding 500 ; setting maximum water height that can be accumilated in a terrace to 500mm
  set daily-infiltration-amount 240 ; setting daily infiltraion rate to 240mm/day
  ;set random-year-indices n-values 30 [random 56] ; Generate 30 random indices from 0 to 55 (inclusive)
  set terrace-creation-period 365
  set evaporation-data []
  set precipitation-data []
  set next-farm-id 0  ; Start the farm IDs from 1
  set runoff-initial-abstraction 3 ; set initial abstraction 3mm
  set last-used-shade 0 ; Starting shade value
  set runoff-cf 1.567
  ;set num-of-terraces-per-season 1 ;; sets each farm to create 1 terrace at 'terrace-creation-period'

  ;;sets labor costs (kg of wheat per dunam) for all agriculture activities
  set build-grapevine-terrace 90
  set build-wheat-field 3.6
  set prepare-harvest-runoff-field 22.5
  set vineyard-planting 34.2
  set vineyard-establishment-phase 35.6
  set vineyard-season-labor 14.4
  set wheat-season-labor 7.2
  ;set grapevine-infrastructure-maintenance 10.1
  ;set wheat-infrastructure-maintenance 0.4
  set wheat-sow-amount 13.4 ; 5 modii per iūgerum which are 13.4kg per dunam (1 modius = 6.7kg; 1 iūgerum is roughly 2.5 dunamns)
  set maintenance-cost 1.1 ; upkeeping annual cost 10%
  set dunam-to-patch-ratio 0.9 ; a patch area in this model is 900 square meters according to the DEM

  ;;sets values of the wine production industry
  set agrisys-annual-wine-production 0
  set agrisys-annual-wine-production-list []
  set wine-to-grape-weight-ratio 0.7

end


to set-climate-random-years
  ; Initialize local lists for holding indices of dry and non-dry years
  let dry-years []
  let non-dry-years []

  ; Iterate over all years in the precipitation-data to sum up each year's precipitation
  ; and categorize them as dry or non-dry based on the drought-threshold
  foreach n-values length precipitation-data [i -> i] [
    i ->
    let year-data item i precipitation-data
    let year-total sum butfirst year-data  ; Calculate total precipitation, skip the first item (year)

    ifelse year-total < drought-threshold [
      set dry-years lput i dry-years
    ] [
      set non-dry-years lput i non-dry-years
    ]
  ]

  ; shuffle each list
  set dry-years shuffle dry-years
  set non-dry-years shuffle non-dry-years


  ; Handle the selection of years based on user input. If consecutive-droughts switch button is TRUE:
ifelse consecutive-droughts [
  ; User requests X consecutive drought years
  let selected-dry-years sublist dry-years 0 (consecutive-droughts-period)
  let total-slots model-run-period  ; Total slots in the simulation period (M)
  let insert-point random (total-slots - length selected-dry-years)  ; Determine where to insert the dry years
  ;let insert-point 3  ; Determine where to insert the dry years <<Debug Mode>>
  print (word "insert-point value is: " insert-point)
  ; Prepare the final list with an appropriate size filled initially with placeholders
  let final-year-sequence n-values total-slots [-1]

  ; Insert dry years at the chosen point
  foreach n-values length selected-dry-years [i -> i] [
    i ->
    set final-year-sequence replace-item (insert-point + i) final-year-sequence (item i selected-dry-years)
  ]

  ; Fill the rest with non-dry years, skipping over already filled slots
  let index 0
  foreach n-values total-slots [i -> i] [
    i ->
    ifelse item i final-year-sequence = -1 [
      ; Check if we still have non-dry years to insert
      if index < length non-dry-years [
        set final-year-sequence replace-item i final-year-sequence (item index non-dry-years)
        set index index + 1
      ]
    ] [ ; Skip if the slot is already filled
    ]
  ]

  set random-year-indices final-year-sequence
] [
  ; If consecutive-droughts switch button is FALSE, select years randomly from all available years
  let all-indices n-values length precipitation-data [i -> i]  ; Recreate list of all indices
  set random-year-indices sublist shuffle all-indices 0 (model-run-period)
  ;print (word "all-indices: " all-indices)
]


  ; Debug prints
  print (word "Dry years indices: " dry-years)
  print (word "Non-Dry years indices: " non-dry-years)
  print (word "Selected years indices: " random-year-indices)
end

; Helper function to calculate the cutoff day index from month and day
to-report calculate-cutoff-day [a-month a-day]
  ; Initialize total days with 0
  let total-days 0
  ; Days in each month, February is always 28
  let days-per-month [31 28 31 30 31 30 31 31 30 31 30 31]

  ; Sum the days for complete months before the end month using foreach
  foreach (range 0 (a-month - 1)) [
    index ->
    set total-days total-days + item index days-per-month
  ]

  ; Add the days from the last, possibly incomplete, month
  print total-days + a-day
  report total-days + a-day
end

; Initializes patch and vineyard variables.
to init-variables
  ; Set initial values for patch attributes.
  ask patches [
    set patch-elevation -1
    set patch-slope-deg  -1
    set patch-slope-pr -1
    set patch-aspect -1
    set patch-face "none"
    set is-catchment-area false
    set is-catchment-border false
    set is-city-area false
    set is-city-border false
    set is-agriculture-terrace false
    set is-potential-wheat-field false
    set patch-terrace-id -1  ; Initialize to false for all patches
    set patch-soil-index "none"
    set patch-runoff-coefficient -1
    set patch-flow-direction -1
    set patch-flow-accumulation -1
    set patch-sub-basin-id -1
    set patch-type "empty"  ; Initialize patch type to 'empty' for all patches
    set coverage-status "natural"  ; Initialize coverage status to 'natural' for all patches
    set patch-runoff-cluster-terrace-id -1
    set stream-id -1
    set is-tributary false
    set is-main-river false
    set runoff-threshold 0
    set patch-runoff-storage 0
    set patch-AMC 1 ; setting initial patch moisture state as dry (AMC = 1)
    set patch-moisture-level [0 0 0  0 0] ; reseting moisture level to empty list
    set patch-rank -1

    ;set attributes related to SCS curve number runoff estimation method
    set CN1S 100
    set CN2 100
    set CN3 100
    set CN2S 100
    set SAMC1 0
    set SAMC2 0
    set IaAMC1 0
    set IaAMC2 0


  ]

  ask terraces [
    ; Set initial values for vineyard attributes.
    set terrace-id -1
    set terrace-sub-basin-id -1
    set terrace-AWS 0
    set annual-terrace-AWS-list []
    set annual-terrace-yield-list []
    set max-ponding 500

  ]


  ask farms [
    ; Set initial values for farms attributes.
    set owned-terraces []
    set owned-vineyards []
    set total-grape-yield-kg 0
    set total-wheat-yield-kg 0
    set wheat-storage 2000 ;starting each far,m with 2,000kg of wheat in their storage. Last couple of years production
    set farm-potential-grapevine-terraces []

  ]
end

; Identifies and marks urban areas on the map.
to set-urban-area
  ; Update patches that are part of the city area
  ask patches with [is-city-area or is-city-border] [
    set patch-type "urban"
    set coverage-status "urban"
    set patch-sub-basin-id -1
  ]

end

; Procedure to classify streams by their stream-order value to minor streams (tributaries) and large streams (manin rivers)
to classify-streams
  ask patches [
    if  stream-order = 4 or stream-order = 5 [
      set is-main-river true
    ]
    if stream-order = 1 or stream-order = 2 or stream-order = 3[
      set is-tributary true
    ]
  ]
end

; Reports all potential terraces based on pre-defined criteria.

to-report find-all-potential-terraces
  report patches with [
    (is-tributary) and
    ;patch-slope-deg <= max-slope-for-terrace and
    is-catchment-area and not is-city-area and
    patch-type = "empty" and
    not is-agriculture-terrace and

    ; Count non-stream, non-urban patches in the same sub-basin, excluding urban patches
    count (patches with [patch-sub-basin-id = [patch-sub-basin-id] of myself and
                        stream-order = -1 and
                        not is-city-area and
                        patch-type != "urban"]) >= runoff-ratio
  ]
end


; Flag all potential wheat agriculture fields as such for later use by farms
to setup-potential-wheat-fields


  ; Criteria 1: Direct neighbors of stream-order 3 or 4, where patch stream-order is -1.
  ask patches with [stream-order = -1 and is-catchment-area ] [
    if any? neighbors4 with [stream-order = 3 or stream-order = 4] [
      set is-potential-wheat-field true
    ]
  ]
  repeat 2 [
    ask patches with [any? neighbors with [stream-order = -1 and is-potential-wheat-field and is-catchment-area ]] [set is-potential-wheat-field true]
  ]

  ask patches with [is-potential-wheat-field and stream-order != -1] [set is-potential-wheat-field false]
  ask patches with [is-city-area] [set is-potential-wheat-field false]
end

to setup-patch-ranks
  ; Make sure all patches have an initial patch-rank value of -1
  ask patches [ set patch-rank -1 ]

  ; Read the entire Ranking.csv file into a list of lists
  let ranking-data csv:from-file "Ranking.csv"

  ; Skip the first item (header) and iterate over each row
  foreach butfirst ranking-data [
    row ->
    ; Extract Xcor, Ycor, and Model Rank from the row
    let xcor-value item 0 row
    let ycor-value item 1 row
    let model-rank item 2 row

    ; Update the patch-rank for the patch at (xcor, ycor)
    ask patches with [pxcor = xcor-value and pycor = ycor-value] [
      set patch-rank model-rank
    ]
  ]
end

to setup-farm-potential-grapevine-terraces
  ask farms [
    ; Reset the farm's list of potential terraces
    set farm-potential-grapevine-terraces []

    ; Manually create a list of patch properties from potential terraces
    let temp-list []
    ask potential-terraces [
      let pt-x pxcor
      let pt-y pycor
      let pt-rank patch-rank
      let pt-incline (patch-slope-pr / 100) ;conver patch slope in precentage-rise units value to fraction between 0 and 1
      let dist distance myself  ; Use 'myself' to refer to the farm in 'ask farms' context
      ;let ratio pt-rank / max(list dist 0.0001)  ; To prevent division by zero
      let ratio (pt-rank / dist) * (1 - pt-incline) ; including all searching parameters for best location [highest rank score): probability of good yield, distance from farm and patch slope
      set temp-list lput (list pt-x pt-y ratio) temp-list
    ]

    ; Sort the list by the ratio in descending order
    set farm-potential-grapevine-terraces sort-by [[a b] -> item 2 a > item 2 b] temp-list
  ]
end

to setup-farm-potential-wheat-fields
  ask farms [
    set farm-potential-wheat-fields []
    let temp-list []

    ask patches with [is-potential-wheat-field and coverage-status = "natural"] [
      let pt-x pxcor
      let pt-y pycor
      let pt-rank patch-rank
      let pt-incline (patch-slope-pr / 100) ;conver patch slope in precentage-rise units value to fraction between 0 and 1
      let dist distance myself  ; 'myself' refers to the farm in the 'ask farms' block

      ; Calculate the selection metric based on distance and patch rank
      let ratio (pt-rank / dist) * (1 - pt-incline) ; including all searching parameters for best location [highest rank score): probability of good yield, distance from farm and patch slope
      set temp-list lput (list pt-x pt-y ratio) temp-list
    ]

    ; Sort potential fields by selection metric in descending order
    set farm-potential-wheat-fields sort-by [[a b] -> item 2 a > item 2 b] temp-list
  ]
end


; Assigns GIS data to patches and sets up various environmental attributes.
to assign-shapefiles-data-to-patches
  print "Load all shapefile to model world started...."
  print date-and-time
  wait 1
  print "Topographic file data (elevation, slopes, aspect and flow data) loading started"
  assign-topographic-data-to-patches ; Assign Shivta area topographic values to patches
  print "Topographic file data loading completed"
  wait 1
  print date-and-time
  print "Patch face (direction: North, South...etc) file data loading started"
  assign-patch-face ; Caluclate compase rose direction from patch-aspect values and set to this attribute
  print "Patch face (direction: North, South...etc) file data loading completed"
  wait 1
  print date-and-time
  print "Catchment area and City area files data loading started"
  set-catchment-area  ; Identify and set catchment area to patches
  set-catchment-border  ; Identify and set catchment border patches
  set-city-area       ; Identify and set city area patches
  set-city-border  ; Identify and set city border patches
  print "Catchment area and City area files data loading completed"
  wait 1
  print date-and-time
  print "Zetan Stream network order file data loading started"
  set-stream-order ; Identify and set Shivta area streams network and streams order (values range 1 to 6; larger value indicates a main river)
  print "Zetan Stream network order file data loading completed"
  wait 1
  print date-and-time
  print "Zetan soil-type file data loading started"
  set-soil-data ;Assign Shivta area soil group type (string) values to patches
  print "Zetan soil-type file data loading completed"
  wait 1
  print date-and-time
  print "Zetan runoff coeeficient calculations started"
  set-runoff-coefficients ;Assign Shivta area runoff coefficient (float) values to patches (based on soil group type of the patch
  print "Zetan runoff coeeficient calculations completed"
  wait 1
  print date-and-time
  print "Zetan sub-basins (IDs) polygons file data loading started"
  set-sub-basin-id ;
  print "Zetan sub-basins (IDs) polygons file data loading completed"
  wait 1
  print date-and-time
  print "Zetan streams network (stream IDs) file data loading started"
  assign-streams-IDs
  print "Zetan streams network (stream IDs) file data loading completed"
  wait 1
  print date-and-time
  print "Load all shapefile to model world completed"
end


;;<<<<<<<<<<<<<<<<< Create agent sets >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
;;<<<<<<<<<<<<<<<<< Create agriculture farms/vineyard mechanism  >>>>>>>>>>>>>>>>>>>>>>>>>>>>



; This procedure constructs a specified number of farms based on suitable base patch coordinates.
to construct-farms
  ; Find potential base patches for farm construction.
  let potential-base-patches-coords find-potential-base-patches-coords

  ; Initialize a list to keep track of coordinates in forbidden areas for farm placement.
  let forbidden-areas-coords []

  ; Counter to keep track of the number of farms created.
  let farms-created 0

  ; Loop to create farms until the desired number is reached or no potential bases are left.
  while [farms-created < number-of-farms and not empty? potential-base-patches-coords] [
    ; Select a random base patch coordinate from the potential list.
    let base-patch-coords one-of potential-base-patches-coords

    ; Find suitable locations for the farm based on the selected base patch and existing forbidden areas.
    let farm-locations find-suitable-farm-locations base-patch-coords forbidden-areas-coords

    ;print (word "Checking farm locations near base patch at: " base-patch-coords) ;<<DEBUG MODE>>

    ; If suitable locations are found, proceed with farm creation.
    ifelse not empty? farm-locations [
      ; Choose a random suitable location for the farm.
      let farm-location one-of farm-locations

      ;print (word "Found suitable farm location at: " farm-location) ;<<DEBUG MODE>>
      ; Create the farm at the selected location.
      ask patch item 0 farm-location item 1 farm-location [
        sprout-farms 1 [
          set shape "house"
          set color green
          set size 3
          set farm-id next-farm-id
          set total-grape-terraces 0
          set total-wheat-terraces 0
          ;set wheat-storage (195 + random 427) ;starting each farm with randomn value of wheat in their storage. Last couple of years production. based on 30,000 verification runs conducted using behaviourSpace interface
          set wheat-storage (750 + random 1251) ;starting each farm with randomn value of wheat in their storage. Last couple of years production.
          set grapes-storage 0 ;starting each farm withkg of grapes in their storage. Accumulated across all years of model run
          set farm-potential-grapevine-terraces []
          set farm-potential-wheat-fields []
          set owned-terraces []
          set owned-vineyards []
          set wheat-failure-records n-values farms-memory-length [random 2] ; Setting some random failures [1] and non-failure [0] wheat annual avg yield records of a farm in a length equal to farms-memory-length variable defined from UI
          set grape-failure-records n-values farms-memory-length [random 2] ; Setting some random failures [1] and non-failure [0] wheat annual avg yield records of a farm in a length equal to farms-memory-length variable defined from UI
          set annual-avg-wheat-yield 0
          set annual-avg-grape-yield 0

        ]
        set patch-type "farm"
        set coverage-status "urban"
        set is-potential-wheat-field false
      ]
      ; Increment the farm ID for the next farm to be created
      set next-farm-id next-farm-id + 1

      ; Increment the count of farms created.
      set farms-created farms-created + 1

      ; Add the farm location to the list of forbidden areas to prevent overlapping farms.
      set forbidden-areas-coords lput farm-location forbidden-areas-coords
    ]  [
      ;print (word "No suitable locations found near base patch at: " base-patch-coords) ;<<DEBUG MODE>>
    ]

    ; Remove the processed base patch coordinates from the list of potential base patches.
    set potential-base-patches-coords remove base-patch-coords potential-base-patches-coords
  ]

  ;print (word "Constructed " farms-created " farms.") ;<<DEBUG MODE>>
end

; Reporter to find potential base patches for farms
; This reporter creates a list of coordinates for patches suitable as base locations for farms
to-report find-potential-base-patches-coords
  let base-patches-coords []
  ask patches with [
    ;patch-slope-deg = 0 and
    patch-type = "empty" and
    stream-order = -1 and
    is-catchment-area = TRUE ;and
    ;any? neighbors with [is-main-river]
  ] [
    set base-patches-coords lput (list pxcor pycor) base-patches-coords
  ]
  ;print (word "Found " length base-patches-coords " potential base patches for farms.") ;<<DEBUG MODE>>
  report base-patches-coords
end

; Reporter to find suitable locations for farms.
; This reporter identifies suitable locations for farms around a given base patch.
; It ensures that locations are within a specified radius and not in forbidden areas.
to-report find-suitable-farm-locations [base-patch-coords forbidden-areas-coords]
  ; Initialize an empty list to store suitable locations.
  let suitable-locations []

  ; Focus on the patch at the given base coordinates.
  ask patch (item 0 base-patch-coords) (item 1 base-patch-coords) [
    ask patches with [
      is-city-area = FALSE and
      stream-order = -1 and
      any? neighbors with [is-tributary] and
      not any? (patches in-radius 1) with [
        ; Ensure the patch is not within 1 units of any forbidden area. (example: another farm location)
        member? (list pxcor pycor) forbidden-areas-coords
      ]
    ] [
      ; Add the current patch coordinates to the list of suitable locations.
      set suitable-locations lput (list pxcor pycor) suitable-locations
    ]
  ]
  ;print (word "Identified " length suitable-locations " suitable farm locations near base patch at: " base-patch-coords) ;<<DEBUG MODE>>
  ; Return the list of suitable locations for farm construction.
  report suitable-locations
end


; Procedure to update the list of terraces owned by each farm.
to update-owned-terraces
  ; Iterate over each farm.
  ask farms [
    let my-farm-id farm-id
    ; Update the list of owned terraces based on the farm's ID.
    set owned-terraces (list-terrace-ids-owned-by my-farm-id)
  ]
end

; Reporter to list terrace IDs owned by a specific farm.
; It returns a list of terrace IDs that are owned by the given farm ID.
to-report list-terrace-ids-owned-by [input-farm-id]
  ; Report the list of terrace IDs where the 'owned-by-farm' attribute matches the input farm ID.
  report [terrace-id] of terraces with [owned-by-farm = input-farm-id]
end

; Procedure to update the list of vineyards owned by each farm.
to update-owned-vineyards
  ; Iterate over each farm.
  ask farms [
    let my-farm-id farm-id

    ; Update the list of owned vineyards. It creates a list of unique vineyard numbers
    ; from terraces owned by this farm, ensuring no duplicates.
    set owned-vineyards remove-duplicates [part-of-vineyard-number] of terraces with [owned-by-farm = my-farm-id]
  ]
end


;;<<<<<<<<<<<<<<<<< Create agriculture terrace/field procedures  >>>>>>>>>>>>>>>>>>>>>>>>>>>>
to evaluate-owned-wheat-terraces
  ask farms [

    ; Determine the amount of wheat available after setting aside the security reserve
    let available-wheat wheat-storage


    ; Iterate over each owned terrace and evaluate its maintenance cost against available wheat
    foreach owned-terraces [
      temp-terrace-id ->
      ; Find the corresponding terrace based on its ID and type
      let this-terrace one-of terraces with [terrace-id = temp-terrace-id and crop-type = "wheat"]
      ; Proceed if the terrace exists
      if this-terrace != nobody [
        ; Calculate the cost for maintaining this wheat terrace
        let terrace-cost 0.5 * dunam-to-patch-ratio * maintenance-cost * (wheat-sow-amount + wheat-season-labor)
        ; Check if there's enough available wheat to cover the cost
        ifelse available-wheat >= terrace-cost [
          ; If so, deduct the cost from available wheat and set the terrace to active
          set available-wheat available-wheat - terrace-cost
          ;print (word "Farm " farm-id "available wheat: " available-wheat)
          ask this-terrace [ set is-active true ]
        ] [
          ; If not, set the terrace to inactive
          ask this-terrace [ set is-active false ]
        ]
      ]
    ]
    ; Update the farm's wheat storage to reflect the remaining wheat after all deductions
    set wheat-storage precision (available-wheat) 1
    ;print (word "Farm " farm-id " wheat storage after deducting owned wheat fields: " precision (wheat-storage) 1)
    ;print ""

  ]
end

to evaluate-owned-grapevine-terraces
  ask farms [

    let current-farm-id farm-id
    let current-grape-terraces-count count terraces with [crop-type = "grapevine" and owned-by-farm = current-farm-id]

    ; Determine the amount of wheat available after setting aside the security reserve
    let available-wheat wheat-storage

    if current-grape-terraces-count > 0 [
    ; Iterate over each owned terrace and evaluate its maintenance cost against available wheat
    foreach owned-terraces [
      temp-terrace-id ->
      ; Find the corresponding terrace based on its ID and type
      let this-terrace one-of terraces with [terrace-id = temp-terrace-id and crop-type = "grapevine"]
      ; Proceed if the terrace exists
      if this-terrace != nobody [
        ; Calculate the cost for maintaining this wheat terrace
        let terrace-cost dunam-to-patch-ratio * maintenance-cost * vineyard-season-labor
        ; Check if there's enough available wheat to cover the cost
        ifelse available-wheat >= terrace-cost [
          ; If so, deduct the cost from available wheat and set the terrace to active
          set available-wheat available-wheat - terrace-cost

          ask this-terrace [ set is-active true ]
        ] [
          ; If not, set the terrace to inactive
          ask this-terrace [ set is-active false ]
        ]
      ]
    ]
    ; Update the farm's wheat storage to reflect the remaining wheat after all deductions
      set wheat-storage precision (available-wheat) 1
      ;print (word "Farm " farm-id " wheat storage after deducting owned grape fields: " precision (wheat-storage) 1)
      ;print ""
    ]
  ]
end


to expand-vineyards
  ask farms [

    setup-farm-potential-grapevine-terraces
    ; Capture the farm-id of the current farm into a local variable
    let current-farm-id farm-id

    let current-grape-terraces-count count terraces with [crop-type = "grapevine" and owned-by-farm = current-farm-id]

    let reserve (wheat-storage * farm-wheat-reserve) ; 10% security, re-evaluated if needed
    ;print (word "Farm " current-farm-id " reserves: " precision (reserve) 1)
    ;print ""

    let available-wheat (wheat-storage - reserve)
    let cost-per-terrace dunam-to-patch-ratio * maintenance-cost * (build-grapevine-terrace + prepare-harvest-runoff-field * runoff-ratio + vineyard-planting + vineyard-establishment-phase + vineyard-season-labor)
    let max-terraces floor (available-wheat / cost-per-terrace)
    ;print (word "Farm " current-farm-id " can create a total of " max-terraces " grape terraces.")
    ;print ""

    if max-terraces > 0 [set color green] ;farms that can expand their vineyards are flaged color green

    repeat max-terraces [
      create-grapevine-terraces farm-id
    ]

    update-owned-terraces
    assign-vineyard-numbers
    update-owned-vineyards

    let new-grape-terraces-count count terraces with [crop-type = "grapevine" and owned-by-farm = current-farm-id]
    ;print (word "Farm " current-farm-id " new-grape-terraces-count: " new-grape-terraces-count)
    ;print (word "Available wheat before deducting GRAPE terraces cost: " precision (available-wheat) 1)
    set available-wheat (available-wheat - ((new-grape-terraces-count - current-grape-terraces-count) * cost-per-terrace))
    ; Adjusted line for clarity and correct context scope
    set wheat-storage precision (available-wheat + reserve) 1
    ;print (word "Farm " FARM-ID " Wheat storage after creating GRAPE terraces without set-aside reserves: " precision (available-wheat) 1)
    ;print""
    ;print (word "Farm " FARM-ID " Wheat storage after all season deductions (+reserves): " precision (wheat-storage) 1)
    ;print ""
  ]
end

to expand-wheat-fields
  ask farms [
    setup-farm-potential-wheat-fields  ; Setup or update the list of potential wheat fields

    ; Capture the farm-id of the current farm into a local variable
    let current-farm-id farm-id
    let current-wheat-fields-count count terraces with [crop-type = "wheat" and owned-by-farm = current-farm-id]

    let reserve (wheat-storage * farm-wheat-reserve) ; Calculating reserve, percentage
    let available-wheat (wheat-storage - reserve)
    let cost-per-field dunam-to-patch-ratio * maintenance-cost * (build-wheat-field + wheat-season-labor + wheat-sow-amount)

    ; Set the number of new wheat fields to create based on the lesser of 3 values:
    ; 1. 10% of the current wheat fields owned by the farm, or
    ; 2. The maximum number of fields affordable within the farm's reserved wheat budget.
    ; 3. 5 units

    let max-fields floor min (list (reserve / cost-per-field) (0.1 * current-wheat-fields-count) (5))

    ;if max-fields > 0 [set color green]  ; Optional: Color farms that can expand their wheat fields green

    if max-fields > 0 [
      ; Call the modified create-wheat-terraces method with the number of fields to create
      create-wheat-terraces current-farm-id max-fields
    ]

    ; Update storage after expenditure
    let new-wheat-fields-count count terraces with [crop-type = "wheat" and owned-by-farm = current-farm-id]
    set available-wheat (available-wheat - ((new-wheat-fields-count - current-wheat-fields-count) * cost-per-field))

    set wheat-storage precision (available-wheat + reserve) 1

    update-owned-terraces
    assign-vineyard-numbers
    update-owned-vineyards
  ]
end

to check-and-create-wheat-terraces
  setup-farm-potential-wheat-fields
  ask farms [
      ;; Each farm calls for terrace creation
    create-wheat-terraces farm-id 1

    ]
    ;; After creating terraces, assign them to vineyards and to farms
  update-owned-terraces
  assign-vineyard-numbers
  update-owned-vineyards

end

;<<<<<<<<<<<<<<<<<<<<new modified wheat terrace code here>>>>>>>>>>>>>>>>>>>>>

to create-wheat-terraces [creator-farm-id number-of-fields]
  let the-farm one-of farms with [farm-id = creator-farm-id]
  let previous-patch nobody

  ask the-farm [
    ; First, create the highest-ranked terrace if any fields are to be created
    ;setup-farm-potential-wheat-fields ; Refresh the potential fields list initially
    let farm-wheat-temp-list farm-potential-wheat-fields

    while [number-of-fields > 0 and not empty? farm-wheat-temp-list] [
      let best-patch first farm-wheat-temp-list
      let target-patch patch (item 0 best-patch) (item 1 best-patch)
      ;let target-patch patch (item 0 best-patch) (item 1 best-patch)

      if [coverage-status] of target-patch = "natural" [
        sprout-neighbour-wheat-terrace target-patch creator-farm-id
        set previous-patch target-patch
        set number-of-fields number-of-fields - 1

        ; Perform updates after creating each terrace
        update-owned-terraces
        assign-vineyard-numbers
        update-owned-vineyards
      ]

      ; Remove the used patch from the temporary list to prevent re-selection
      set farm-wheat-temp-list remove-item 0 farm-wheat-temp-list

      if number-of-fields > 0 and previous-patch != nobody [
        ask previous-patch [
          ; Move the access of farm-specific data outside of the patch context or ensure it's handled correctly
          ;let farm-potential-wheat-fields-list []
          ;ask the-farm [
          ;  set farm-potential-wheat-fields-list farm-potential-wheat-fields
          ;]
          let adjacent-patches neighbors with [
            ;is-potential-wheat-field and
            not any? terraces with [patch-terrace-id = [patch-terrace-id] of self and crop-type = "wheat"] and
            member? self farm-wheat-temp-list  ; Ensure it is executed in the correct context
          ]
          if any? adjacent-patches [
            let next-patch max-one-of adjacent-patches [patch-rank]  ; Select the highest ranked patch
            sprout-neighbour-wheat-terrace next-patch creator-farm-id
            set previous-patch next-patch
            set number-of-fields number-of-fields - 1

            ; Move the updates to the farm level
            ask the-farm [
              update-owned-terraces
              assign-vineyard-numbers
              update-owned-vineyards
              ;setup-farm-potential-wheat-fields
            ]
            ; Remove the used patch from the temporary list to prevent re-selection
            set farm-wheat-temp-list remove-item (position next-patch farm-wheat-temp-list) farm-wheat-temp-list

          ]
        ]
      ]
    ]
  ]
end

to sprout-neighbour-wheat-terrace [next-patch creator-farm-id]
  ask next-patch [
    sprout-terraces 1 [
      set shape "wheat"
      set max-ponding 200
      set size 2
      set heading patch-aspect
      set terrace-id next-terrace-id
      set terrace-sub-basin-id [patch-sub-basin-id] of self
      set owned-by-farm creator-farm-id
      set crop-type "wheat"
      set terrace-ponding-water 0
      set annual-terrace-AWS-list []
      set annual-terrace-yield-list []
      set annual-terrace-yield-efficiency []
      set is-active true
    ]
    set patch-type "wheat"
    set patch-terrace-id next-terrace-id
    set coverage-status "wheat"
    set is-agriculture-terrace true
  ]
  set next-terrace-id next-terrace-id + 1
end

;<<<<<<<<<<<<<<<<<<<<wheat terrace code above this line>>>>>>>>>>>>>>>>>>>>>

to create-grapevine-terraces [creator-farm-id]
  let terrace-created false
  let the-farm one-of farms with [farm-id = creator-farm-id]

  ; Before attempting to create terraces, ensure the farm's potential terrace list is updated
  ask the-farm [
    let current-list farm-potential-grapevine-terraces
    let updated-list []

    ; Iterate through the current list and filter out patches that are no longer available
    foreach current-list [
      potential-patch ->
      let x item 0 potential-patch
      let y item 1 potential-patch
      let patch-at-coords patch x y

      ; Check if the patch is still available for terrace creation
      if [coverage-status] of patch-at-coords != "vineyard" [
        set updated-list lput potential-patch updated-list
      ]
    ]

    set farm-potential-grapevine-terraces updated-list
  ]


  if the-farm != nobody [
    let farm-list [farm-potential-grapevine-terraces] of the-farm

    while [not terrace-created and not empty? farm-list] [
      ; Get the first sublist which represents the highest (patch-rank / distance) ratio
      let current-sublist first farm-list
      let current-patch patch (item 0 current-sublist) (item 1 current-sublist)

      if current-patch != nobody [
        if has-sufficient-runoff-cluster? current-patch [
          ; If the current patch passes the has-sufficient-runoff-cluster? check, attempt to create a terrace
          let chosen-crop-type "grapevine"
          set terrace-created setup-terrace current-patch creator-farm-id chosen-crop-type
          if terrace-created [
            ; If a terrace is successfully created, remove this patch from the global potential-terraces to prevent future attempts on it by any farm
            set potential-terraces potential-terraces with [self != current-patch]
          ]
        ]
        ; Remove the current sublist from the farm-specific list regardless of terrace creation outcome
        set farm-list butfirst farm-list
      ]
    ]

    ; Update the farm's list after processing
    ask the-farm [ set farm-potential-grapevine-terraces farm-list ]

    ifelse not terrace-created [
      print "No suitable patch found for creating a grapevine terrace."
    ] [
      set global-terrace-created true
    ]
  ]
end


; Helper procedure to check if there are enough suitable runoff patches for a terrace.
to-report has-sufficient-runoff-cluster? [selected-patch]
  ;print (word "Checking for sufficient runoff cluster at: " selected-patch) ;<<DEBUG MODE>>
  ; Identify potential runoff patches around the selected patch.
  let potential-runoff-patches find-potential-runoff-patches selected-patch

  ; Report true if the number of potential runoff patches meets the required ratio.
  report (count potential-runoff-patches) >= runoff-ratio
end


; Procedure to identify potential runoff patches for a given patch in the same sub-basin.
to-report find-potential-runoff-patches [selected-patch]
  ; Retrieve the sub-basin ID of the selected patch.
  let sub-basin-id [patch-sub-basin-id] of selected-patch

  ; Report patches within the same sub-basin that meet specific runoff criteria.
  report patches with [
    patch-sub-basin-id = sub-basin-id and
    stream-order = -1 and
    patch-runoff-cluster-terrace-id = -1 and
    coverage-status = "natural" and
    not is-potential-wheat-field  ; Exclude potential wheat field patches
  ]
end

to-report setup-terrace [suitable-patch creator-farm-id chosen-crop-type]
  let potential-runoff-patches find-potential-runoff-patches suitable-patch
  let runoff-cluster create-runoff-cluster potential-runoff-patches suitable-patch

  ifelse count runoff-cluster >= runoff-ratio [
    ask suitable-patch [
      sprout-terraces 1 [
        if chosen-crop-type = "grapevine" [
          set shape "vineyard"
          set max-ponding 500
          set is-active true

        ]
        if chosen-crop-type = "wheat" [
          set shape "wheat"
          set max-ponding 200
          set is-active true
        ]
        if chosen-crop-type = "none" [
          set shape "empty-terrace"
          set max-ponding 0
          set is-active false
        ]
        set size 2
        set heading patch-aspect
        ;set color green
        set terrace-id next-terrace-id
        set terrace-sub-basin-id [patch-sub-basin-id] of suitable-patch
        set owned-by-farm creator-farm-id
        set terrace-ponding-water 0
        set crop-type chosen-crop-type ; Set the crop type here
        set annual-terrace-AWS-list []
        set annual-terrace-yield-list []
        set annual-terrace-yield-efficiency []
        ;set is-active true

      ]
      set patch-type "vineyard"
      set patch-terrace-id next-terrace-id
      set coverage-status "vineyard"
      set is-agriculture-terrace true
    ]
    set next-terrace-id next-terrace-id + 1
    ask runoff-cluster [
      set coverage-status "clear with mounds"
      set patch-runoff-cluster-terrace-id [patch-terrace-id] of suitable-patch
    ]
    report true
  ] [
    report false
  ]
end


; Procedure to create a cluster of runoff patches around a suitable patch.
to-report create-runoff-cluster [potential-runoff-patches suitable-patch]
  ;print "Creating runoff cluster" ;<<DEBUG MODE>>
  ; Initialize an empty patch set for the runoff cluster.
  let runoff-cluster no-patches

  ; Retrieve the sub-basin ID of the suitable patch for consistency in runoff patch selection.
  let sub-basin-id [patch-sub-basin-id] of suitable-patch

  ; Filter potential patches in the same sub-basin and ensure they meet other runoff criteria.
  let potential-sub-basin-patches potential-runoff-patches with [
    patch-sub-basin-id = sub-basin-id and
    stream-order = -1 and
    patch-runoff-cluster-terrace-id = -1 and
    coverage-status = "natural"
  ]

  ; Start the runoff cluster with the closest patch to the suitable patch.
  if any? potential-sub-basin-patches [
    let initial-patch min-one-of potential-sub-basin-patches [distance suitable-patch]
    set runoff-cluster (patch-set runoff-cluster initial-patch)
    ; Remove the initial patch from potential patches to avoid duplication.
    set potential-sub-basin-patches potential-sub-basin-patches with [self != initial-patch]
  ]

  ; Iteratively add neighboring patches to the runoff cluster.
  let continue-loop true
  while [(count runoff-cluster < runoff-ratio) and (any? potential-sub-basin-patches) and continue-loop] [
    let new-neighbors no-patches

    ; Find neighboring patches that are part of the potential runoff patches.
    foreach sort runoff-cluster [
      this-patch ->
      ask this-patch [
        ask neighbors [
          if (member? self potential-sub-basin-patches and not member? self runoff-cluster) [
            set new-neighbors (patch-set new-neighbors self)
          ]
        ]
      ]
    ]

    ; Add new neighbors to the runoff cluster or break loop if no more neighbors can be added.
    ifelse any? new-neighbors [
      foreach sort new-neighbors [
        this-patch ->
        if (count runoff-cluster < runoff-ratio) [
          set runoff-cluster (patch-set runoff-cluster this-patch)
          set potential-sub-basin-patches potential-sub-basin-patches with [self != this-patch]
        ]
      ]
    ] [
      ;print "No more neighbors to add, exiting loop" ;<<DEBUG MODE>>
      ; Break the loop if no more suitable neighbors are available to add.
      set continue-loop false
    ]
  ]

  ;print "Runoff cluster created" ;<<DEBUG MODE>>
  ; Report the final runoff cluster.
  report runoff-cluster
end


; Procedure to assign vineyard numbers to terraces, grouping them into distinct vineyards.
to assign-vineyard-numbers
  ; Start with the first vineyard number.
  let current-vineyard-number 1

  ; Initially, set all terraces to have no associated vineyard number.
  ask terraces [
    set part-of-vineyard-number 0
  ]

  ; Assign a unique vineyard number to each connected group of terraces.
  while [any? terraces with [part-of-vineyard-number = 0]] [
    ; Select a starting terrace to begin grouping.
    let seed-terrace one-of terraces with [part-of-vineyard-number = 0]
    let same-farm-and-stream-terraces terraces with [
      owned-by-farm = [owned-by-farm] of seed-terrace and
      stream-id = [stream-id] of seed-terrace
    ]

    ; Find all terraces connected to the seed terrace and assign the current vineyard number.
    let connected-terraces find-connected-terraces seed-terrace same-farm-and-stream-terraces
    ask connected-terraces [
      set part-of-vineyard-number current-vineyard-number
    ]

    ; Increment the vineyard number for the next group of terraces.up
    set current-vineyard-number current-vineyard-number + 1
  ]
end

; Helper function to find all terraces connected to a given start terrace.
to-report find-connected-terraces [start-terrace same-farm-and-stream-terraces]
  ; Initialize with the start terrace as the first connected terrace.
  let connected-terraces turtle-set start-terrace

  ; Set to keep track of terraces to check for connections.
  let check-terraces turtle-set start-terrace
  let new-found true

  ; Loop to find all connected terraces.
  while [new-found] [
    set new-found false
    let neighbors-to-check no-turtles

    ; Identify neighboring terraces that are part of the same farm and stream.
    ask check-terraces [
      let nearby-terraces same-farm-and-stream-terraces in-radius 2 with [not member? self connected-terraces]
      set neighbors-to-check (turtle-set neighbors-to-check nearby-terraces)
    ]

    ; Add the new neighbors to the set of connected terraces.
    if any? neighbors-to-check [
      set connected-terraces (turtle-set connected-terraces neighbors-to-check)
      set check-terraces neighbors-to-check
      set new-found true
    ]
  ]

  ; Report the set of all connected terraces.
  report connected-terraces
end


;;<<<<<<<<<<<<<<<<<<<<<<<<<< Terrace and Crops Water-Balance Methods Section>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

to call-runoff-flow
   if daily-rainfall-amount > 0 [
    add-direct-rainfall-to-terrace
    create-runoff ;setup runoff agents with runoff-amount 0
    calculate-runoff  ;; This sets the runoff-amount for the runoff agent
  ]

  update-antecedent-rainfall ;;set patches moisture state for next tick runoff calculation


  if any? runoffs [
    ask runoffs with [runoff-amount = 0] [
      die  ;; Remove the agent if the condition is met
    ]

    update-runoff
  ]
  ask terraces with [ terrace-ponding-water > 0] [
    update-water-balance
    ]
end

to add-direct-rainfall-to-terrace
  ask terraces [
    let rainfall-to-add daily-rainfall-amount
    ifelse (terrace-ponding-water + rainfall-to-add <= max-ponding) [
      set terrace-ponding-water terrace-ponding-water + rainfall-to-add
    ]  [
      let excess-rainfall (terrace-ponding-water + rainfall-to-add - max-ponding)
      set terrace-ponding-water max-ponding
      ;; Create runoff agent for the excess rainfall
      ask patch-here [
        sprout-runoffs 1 [
          set runoff-amount excess-rainfall
          set color blue  ;; Just as an example to visually differentiate runoff agents
          set shape "circle"
          set size 0.5
        ]
      ]
    ]
  ]
end



to update-runoff
  ;; Main method called from the go procedure to handle runoff logic

  ;; Process the runoff agents based on their current patch's coverage-status until no more runoffs remain
  while [any? runoffs] [
    ask runoffs [
      let current-patch patch-here
      let coverage [coverage-status] of current-patch
      let stream-order-val [stream-order] of current-patch

      ;; If the runoff is in a stream of order 3 or 4 and near a wheat field, transfer its runoff amount to the wheat field cluster - divid each runoff agent amount evenly among all wheat terraces (fields) of same cluster (ID)
      if ((stream-order-val = 3 or stream-order-val = 4) and coverage != "vineyard") [

        ;; Find neighboring patches that are wheat terraces
        let neighboring-wheat-patches neighbors with [any? terraces-here with [crop-type = "wheat"]]
        if any? neighboring-wheat-patches [
          ;; For each neighboring wheat patch, find its terraces
          let wheat-terraces (turtle-set)
          ask neighboring-wheat-patches [
            ask terraces-here with [crop-type = "wheat"] [
              set wheat-terraces (turtle-set wheat-terraces self)
            ]
          ]
          if any? wheat-terraces [
            ;; Randomly select one terrace to determine the vineyard-id
            let sample-terrace one-of wheat-terraces
            let vineyard-id [part-of-vineyard-number] of sample-terrace

            ;; Find all terraces in the same vineyard cluster
            let cluster-terraces terraces with [part-of-vineyard-number = vineyard-id and crop-type = "wheat"]

            ;; Distribute runoff
            let total-runoff-amount [runoff-amount] of self
            let each-terrace-runoff total-runoff-amount / count cluster-terraces

            ask cluster-terraces [
              let possible-addition min (list each-terrace-runoff (max-ponding - terrace-ponding-water))
              set terrace-ponding-water terrace-ponding-water + possible-addition
              set total-runoff-amount total-runoff-amount - possible-addition
            ]

            ifelse total-runoff-amount > 0 [
              set runoff-amount total-runoff-amount
              move-runoff-agent self
            ]  [
              die
            ]
          ]
        ]
      ]


     ;; If the runoff is on a terrace of grapes, transfer its runoff amount to the terrace
      if coverage = "vineyard" [
        let current-terrace one-of terraces with [terrace-id = [patch-terrace-id] of current-patch]
        if current-terrace != nobody [
          let incoming-runoff-amount [runoff-amount] of self
          let excess-water add-runoff-to-terrace self current-terrace incoming-runoff-amount
          ifelse excess-water > 0 [
            set runoff-amount excess-water
            move-runoff-agent self
          ] [
            die  ;; Remove the agent if no excess water
          ]
        ]
      ]


      ;; If the runoff is on a terrace of wheat (wheat field), transfer its runoff amount to the terrace
      if coverage = "wheat" [
        let current-terrace one-of terraces with [terrace-id = [patch-terrace-id] of current-patch]
        ;print (word "current terrace id command excuted")
        if current-terrace != nobody [
          ;print (word "current terrace is not nobody. if condition met")
          let incoming-runoff-amount [runoff-amount] of self
          ;print (word "current incoming-runoff-amount is: " incoming-runoff-amount )
          let excess-water add-runoff-to-terrace self current-terrace incoming-runoff-amount
          ;print (word "current excess-water: " excess-water)
          ifelse excess-water > 0 [
            set runoff-amount excess-water
            ;print (word " excess-water > 0 and runoff amount set as such")
            move-runoff-agent self
            ;print (word " move runoff over wheat terrace excuted")
          ] [
            die  ;; Remove the agent if no excess water
            ;print (word " runoff over wheat filed died")
          ]
        ]
      ]

      ;; Handle runoff for patches associated with a terrace (clear with mounds coverage)
      if coverage = "clear with mounds" [
        let associated-terrace one-of terraces with [terrace-id = [patch-runoff-cluster-terrace-id] of current-patch]
        if associated-terrace != nobody [
          ifelse [terrace-ponding-water] of associated-terrace < [max-ponding] of associated-terrace [
            move-to associated-terrace
          ] [
            move-runoff-agent self
          ]
        ]
      ]

      ;; Move the runoff agent according to flow direction for urban or natural patches
      if coverage = "urban" or coverage = "natural" [
        move-runoff-agent self  ; 'self' refers to the current runoff agent
      ]


    ]
  ]
end

to create-runoff
  ;; Adjusted to avoid creating runoff agents on terraces where daily rainfall has been directly added.
  ask patches with [is-catchment-area] [
    ;; Only create runoff if the patch is not an agricultural terrace, or if it's a terrace that might still need runoff handling.
    if not any? terraces-here [
      sprout-runoffs 1 [
        set size 0.5
        set color blue
        set shape "circle"
        set runoff-amount 0  ;; This will be set in the calculate-runoff method.
      ]
    ]
  ]
end


to-report add-runoff-to-terrace [runoff-agent terrace-agent incoming-runoff-amount]
  ;; Retrieve the max-ponding capacity directly from the terrace-agent passed as an argument
  let temp_max_ponding [max-ponding] of terrace-agent
  let temp_terrace_ponding_water [terrace-ponding-water] of terrace-agent

  ;; Check if adding the runoff to the terrace would exceed the terrace's max-ponding capacity
  ifelse (temp_terrace_ponding_water + incoming-runoff-amount > temp_max_ponding) [
    ;; Calculate the excess water if the capacity is exceeded
    let excess-water incoming-runoff-amount - (temp_max_ponding - temp_terrace_ponding_water)
    ask terrace-agent [
      set terrace-ponding-water temp_max_ponding  ;; Set to max capacity
    ]
    report excess-water  ;; Return the excess water amount
  ] [
    ;; If not exceeding, add the runoff amount to the terrace's ponding water
    ask terrace-agent [
      set terrace-ponding-water temp_terrace_ponding_water + incoming-runoff-amount
    ]
    report 0  ;; Indicate no excess water
  ]
end


to set-patches-curve-number-index
  ; set patch CN2 (for normal soil moisture state) index value based on patch soil index (soil group)
  ask patches with [patch-soil-index = "R1" or patch-soil-index = "R4"] [set CN2 92]
  ask patches with [patch-soil-index = "S1" or patch-soil-index = "S2" or patch-soil-index = "S3"] [set CN2 85]
  ask patches with [patch-soil-index = "X6"] [set CN2 77]
  ;ask patches with [coverage-status = "urban"] [set CN2 98]
  ask patches [
    set CN3 min(list 100 round(CN2 / (0.427 + 0.00573 * CN2))) ; set patch CN3 (for wet soil moisture state) index value
    let SPR (patch-slope-pr / 100)
    let term1 (CN3 - CN2)
    let term2 (0.5 - 0.714 * exp(-7.125 * SPR))
    set CN2S min(list 100 round ( term2 * term1 + CN2)) ; set patch CN2S (for normal soil moisture state) index value and adjusted for patch slope
    set CN1S min(list 100 round (CN2S / (2.281 - 0.01281 * CN2S))) ; set patch CN1 (for dry soil moisture state) index value with slope adjustments (based on patch CN2S value)
    set SAMC1 1.42 * ((25400 / CN1S) - 254) ; converting S value to lambda 0.05 ratio approch
    set SAMC2 1.42 * ((25400 / CN2S) - 254) ; converting S value to lambda 0.05 ratio approch
    set IaAMC1 0.05 * SAMC1 ; set patch initial abstraction for dry soil moisture state (based on lambda 0.05 ratio approch)
    set IaAMC2 0.05 * SAMC2 ; set patch initial abstraction for wet soil moisture state (based on lambda 0.05 ratio approch)
  ]
end


to calculate-runoff
  ask runoffs [
    let current-patch patch-here  ; Get the patch where the runoff agent is located

    ; Set runoff-amount for vineyard or urban patches
    ;if ([coverage-status] of current-patch = "vineyard" or [coverage-status] of current-patch = "wheat" or [coverage-status] of current-patch = "urban") [
    if ([coverage-status] of current-patch = "urban") [
      set runoff-amount daily-rainfall-amount
    ]

    ; Set runoff-amount for natural or clear with mounds patches
    if ([coverage-status] of current-patch = "natural" or [coverage-status] of current-patch = "clear with mounds") [
      ; Check patch-AMC and calculate runoff-amount accordingly
      if ([patch-AMC] of current-patch = 1) [
        ; Check if daily rainfall is greater than IaAMC1
        ifelse (daily-rainfall-amount > [IaAMC1] of current-patch) [
          let S1 ([SAMC1] of current-patch)
          set runoff-amount (((daily-rainfall-amount - 0.05 * S1) ^ 2) / (daily-rainfall-amount + 0.95 * S1)) * (runoff-cf)
        ] [
          set runoff-amount 0
        ]
      ]

      if ([patch-AMC] of current-patch = 2) [
        ; Check if daily rainfall is greater than IaAMC2
        ifelse (daily-rainfall-amount > [IaAMC2] of current-patch) [
          let S2 ([SAMC2] of current-patch)
          set runoff-amount (((daily-rainfall-amount - 0.05 * S2) ^ 2) / (daily-rainfall-amount + 0.95 * S2)) * (runoff-cf)
        ] [
          set runoff-amount 0
        ]

      ]
    ]
  ]

  ; Increasing runoff amount of terrace runoff-ratio patches by 30% (based on Avdat farm experiment data)
  ask runoffs [
    let current-patch patch-here  ; Get the patch where the runoff agent is located

    if ([coverage-status] of current-patch = "clear with mounds") [
      set runoff-amount 1.3 * runoff-amount
    ]

  ]
  ;ask runoffs with [runoff-amount = 0] [die] ; remove any runoff agent that has not generated any runoff amount
end


to move-runoff-agent [runoff-agent]
  ask runoff-agent [
    ; Set the agent's heading based on the current patch's flow direction
    let current-heading flow-direction-to-heading [patch-flow-direction] of patch-here
    set heading current-heading

    ; Move the agent one step forward in the new heading direction
    let target-patch patch-ahead 1
    ifelse target-patch != nobody [
      move-to target-patch


      ; Runoff agent removed from model if outside watershed area
      if [not is-catchment-area] of target-patch [
        die
      ]
    ] [
      die ; Runoff agent removed from model if there is no patch ahead (e.g., at the world's edge)
    ]
  ]
end


; Helper function to convert flow direction to heading
to-report flow-direction-to-heading [flow-direction]
  ; Use ifelse statements to determine the heading based on the flow direction
  ifelse flow-direction = 64 [
    report 0
  ] [
    ifelse flow-direction = 128 [
      report 45
    ] [
      ifelse flow-direction = 1 [
        report 90
      ] [
        ifelse flow-direction = 2 [
          report 135
        ] [
          ifelse flow-direction = 4 [
            report 180
          ] [
            ifelse flow-direction = 8 [
              report 225
            ] [
              ifelse flow-direction = 16 [
                report 270
              ] [
                ifelse flow-direction = 32 [
                  report 315
                ] [
                  ; Default case if flow-direction doesn't match any expected value
                  report 0
                ]
              ]
            ]
          ]
        ]
      ]
    ]
  ]
end

to update-water-balance
  ask terraces with [terrace-ponding-water > 0] [
    ifelse terrace-ponding-water > daily-infiltration-amount [
      ; If ponding water is greater than daily infiltration amount
      set terrace-AWS precision (terrace-AWS + daily-infiltration-amount) 1
      set terrace-ponding-water precision (terrace-ponding-water - daily-infiltration-amount) 1
    ] [
      ; If ponding water is less than or equal to daily infiltration amount
      set terrace-AWS precision (terrace-AWS + terrace-ponding-water) 1
      set terrace-ponding-water 0
    ]
  ]
end


;;<<<<<<<<<<<<<<<<<<<<<<<<< Terrace Yield Section >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


; Procedure to calculate and update terrace yields at the end of each year
to terrace-yield
  ask terraces [
    ; Ensure that there is at least one entry in the annual-terrace-AWS-list
    if not empty? annual-terrace-AWS-list [
      ; Get the most recent year's terrace-AWS value
      let last-aws last annual-terrace-AWS-list
      let yield 0

      ; Calculate yield for grapevines
      if crop-type = "grapevine" and is-active[
        if last-aws >= 200 and last-aws <= 1000 [
          ;; Multiply by 0.9 to account for modern pest control componnet of the yield formula. Multiply by 0.09 to calculate for terrace size of 900 sqr meters (from 1 hectare)
          set yield precision (((-0.0144 * (last-aws ^ 2) + (16.931 * last-aws) + 340.68) * 0.9 ) * 0.09) 1
        ]
      ]

      ; Calculate yield for wheat
      if crop-type = "wheat" and is-active [
        if last-aws >= 80 and last-aws <= 2150 [
          ;; Multiply by 0.9 to account for modern pest control componnet of the yield formula. Multiply by 0.09 to calculate for terrace size of 900 sqr meters (from 1 hectare)
          ;; Multiply by 0.5 accounts for fallowing wheat fields. Each alternate year only half the field is saw and the other half is left to rest, and vice versa the next year.
          ;; Thus each year the total yield is only half of what it could be for same amount of water on a whole field size lot.
          set yield precision ((((-0.0023 * (last-aws ^ 2) + (5.1382 * last-aws) - 414.47) * 0.9 ) * 0.09) * 0.5) 1
        ]
      ]

      ; Ensure yield is not negative
      if yield < 0 [ set yield 0 ]

      ; Append the calculated yield to the terrace's annual yield list
      set annual-terrace-yield-list lput yield annual-terrace-yield-list
    ]
  ]
end




to update-terrace-total-yields
  ask terraces [
    ; Calculate the total yield by summing up all items in the annual-terrace-yield-list
    set total-terrace-yield precision (sum annual-terrace-yield-list) 1

    ; Check if the annual-terrace-yield-list is not empty before calculating the average
    ifelse length annual-terrace-yield-list > 0 [
      set average-terrace-yield precision ((total-terrace-yield / length annual-terrace-yield-list)) 1
    ]  [
      ; If the list is empty, set the average yield to 0
      set average-terrace-yield 0
    ]
  ]
end

to update-annual-terrace-yield-efficiency
  ask terraces [
    let last-yield last annual-terrace-yield-list
    ; Calculate efficiency based on crop type and append it to the list
    let efficiency 0
    if crop-type = "wheat" [
      ;; Dividing actual yield of terrace by wheat maximum possible yield for terrace area according to wheat yield formula (after adjustment to terraces size, pest control and fallow alternate years)
      set efficiency precision (last-yield / 103.5) 3
    ]
    if crop-type = "grapevine" [
      ;; Dividing actual yield of terrace by grapevine maximum possible yield for terrace area according to grapevine yield formula (after adjustment to terraces size and pest control
      set efficiency precision (last-yield / 430.7) 3
    ]
    set annual-terrace-yield-efficiency lput efficiency annual-terrace-yield-efficiency
  ]
end


to calculate-farm-yields
  ; Iterate over each farm
  ask farms [
    ; Initialize the total yields and terrace counts to 0
    let total-grape-yield 0
    let total-wheat-yield 0
    let grape-terrace-count 0
    let wheat-terrace-count 0

    set current-season-wheat-yield-kg 0 ; Ensure it's reset at the start of yield calculation

    ; Iterate over each owned terrace
    foreach owned-terraces [
      temp-terrace-id ->
      ; Find the terrace with the given terrace-id
      let this-terrace one-of terraces with [terrace-id = temp-terrace-id]

      ; Check the crop type, add its yield to the appropriate total, and count the terraces
      if this-terrace != nobody [
        if [crop-type] of this-terrace = "grapevine" [
          set total-grape-yield total-grape-yield + [total-terrace-yield] of this-terrace
          set grape-terrace-count grape-terrace-count + 1
        ]
        if [crop-type] of this-terrace = "wheat" [
          set total-wheat-yield total-wheat-yield + [total-terrace-yield] of this-terrace

          ; First, retrieve the list and then get the last element from it
          let terrace-yield-list [annual-terrace-yield-list] of this-terrace
          let this-years-yield ifelse-value (not empty? terrace-yield-list) [last terrace-yield-list] [0]
          set current-season-wheat-yield-kg precision (current-season-wheat-yield-kg + this-years-yield) 1

          set wheat-terrace-count wheat-terrace-count + 1
        ]
      ]
    ]

    ; Update the farm's total yield and terrace count variables
    set total-grape-yield-kg precision (total-grape-yield) 0
    set total-wheat-yield-kg precision (total-wheat-yield) 0
    set total-grape-terraces grape-terrace-count
    set total-wheat-terraces wheat-terrace-count

    ; Update wheat-storage with the current season's wheat yield
    set wheat-storage precision (wheat-storage + current-season-wheat-yield-kg) 1
    ;print (word "Statistics for model year run: " (current-year-index + 1967))
    ;print ""
    ;print (word "Farm " farm-id " wheat storage at season end: " precision (wheat-storage) 1)
    ;print ""
  ]
end

to update-farm-annual-average-yields
  ask farms [
    ; Initialize sum variables and counters
    let total-wheat-yield 0.0
    let wheat-count 0
    let total-grape-yield 0.0
    let grape-count 0

    ; Go through each owned terrace ID and aggregate yields if the terrace is active and of the correct type
    foreach owned-terraces [
      this-terrace-id ->
      let current-terrace one-of terraces with [terrace-id = this-terrace-id and is-active]

      ; Proceed only if a valid terrace is found
      if current-terrace != nobody [
        ; Check crop type and accumulate yield
        if [crop-type] of current-terrace = "wheat" [
          set total-wheat-yield total-wheat-yield + last [annual-terrace-yield-list] of current-terrace
          set wheat-count wheat-count + 1
        ]
        if [crop-type] of current-terrace = "grapevine" [
          set total-grape-yield total-grape-yield + last [annual-terrace-yield-list] of current-terrace
          set grape-count grape-count + 1
        ]
      ]
    ]

    ; Calculate and update average yields if there are active terraces
    if wheat-count > 0 [
      set annual-avg-wheat-yield precision (total-wheat-yield / wheat-count) 1
    ]
    if grape-count > 0 [
      set annual-avg-grape-yield precision (total-grape-yield / grape-count) 1
    ]
  ]
end




; Procedure to calculate the total annual production of wine in liters from grapevine terraces
to calculate-total-wine-production
  ; Reset the aggregate grape yield for the new calculation
  let aggregate-grape-yield 0


  ; Instead of iterating over farms, iterate directly over terraces with grapevines
  ask terraces with [crop-type = "grapevine"] [
    if not empty? annual-terrace-yield-list [
      ; Add the most recent yield of each grapevine terrace to the aggregate yield
      set aggregate-grape-yield aggregate-grape-yield + last annual-terrace-yield-list
    ]
  ]

  ; Convert the updated aggregate grape yield into liters of wine (assuming 0.7 liters of wine per kg of grapes)
  set  agrisys-annual-wine-production precision (aggregate-grape-yield * wine-to-grape-weight-ratio) 1

  ; populate global list of years and wine production
  set  agrisys-annual-wine-production-list lput (list (year - 535) agrisys-annual-wine-production) agrisys-annual-wine-production-list


  ; Report or display the total wine production for the current year
  if (day = 31 and current-month = 12) [
    ;print (word "Total wine production for the current year " year " (in liters): "  agrisys-annual-wine-production)
    ;print (word "Overall agrisystem list (pairs): " agrisys-annual-wine-production-list )
  ]


end


;;<<<<<<<<<<<<<<<<<<<<<<<<<< Precipitation Data Methods Section >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
to load-precipitation-data
  ; Load precipitation data from the CSV file and handle errors if the file is missing
  ifelse file-exists? "Precipitation Data.csv" [
    set precipitation-data csv:from-file "Precipitation Data.csv"
  ] [
    user-message "Error: Precipitation Data CSV file is missing. Please ensure it's in the correct directory."
    stop
  ]
end

to reconstruct-precipitation-data
  ; Initialize the new variable for seasonal precipitation data
  let seasonal-precipitation-data []
  let cutoff-index calculate-cutoff-day season-start-month season-start-day

  ; Loop through all but the last year of precipitation data
  foreach (range (length precipitation-data - 1)) [i ->
    let current-year item i precipitation-data
    let next-year item (i + 1) precipitation-data

    ; Combine the part from August 1st to December 31st of the current year
    ; with January 1st to July 31st of the next year
    let start-of-season sublist current-year cutoff-index 366
    let end-of-season sublist next-year 1 (cutoff-index)

    ; Create a new year list with combined data
    let new-year-list fput i (sentence start-of-season end-of-season)
    set seasonal-precipitation-data lput new-year-list seasonal-precipitation-data
  ]

  ; Handle the last year by connecting it to the first year
  let last-year item (length precipitation-data - 1) precipitation-data
  let first-year item 0 precipitation-data
  let last-start-of-season sublist last-year cutoff-index 366
  ;let first-end-of-season sublist first-year 2 (cutoff-index + 1)
  let first-end-of-season sublist first-year 1 (cutoff-index)
  let last-year-list fput (length precipitation-data - 1) (sentence last-start-of-season first-end-of-season)
  set seasonal-precipitation-data lput last-year-list seasonal-precipitation-data

  ; Export the seasonal precipitation data to a CSV file
  ;csv:to-file "reconstructed precipitation.csv" seasonal-precipitation-data

  ; Update global precipitation-data with the newly constructed seasonal data
  set precipitation-data seasonal-precipitation-data
end



to load-evaporation-data
  ; Load evaporation data from the CSV file and handle errors if the file is missing
  ifelse file-exists? "Evaporation Data.csv" [
    set evaporation-data csv:from-file "Evaporation Data.csv"
  ] [
    user-message "Error: Evaporation Data CSV file is missing. Please ensure it's in the correct directory."
    stop
  ]
end

to reconstruct-evaporation-data
  ; Initialize the new variable for seasonal precipitation data
  let seasonal-evaporation-data []
  let cutoff-index calculate-cutoff-day season-start-month season-start-day

  ; Loop through all but the last year of precipitation data
  foreach (range (length evaporation-data - 1)) [i ->
    let current-year item i evaporation-data
    let next-year item (i + 1) evaporation-data

    ; Combine the part from August 1st to December 31st of the current year
    ; with January 1st to July 31st of the next year
    let start-of-season sublist current-year cutoff-index 366
    let end-of-season sublist next-year 1 (cutoff-index)

    ; Create a new year list with combined data
    let new-year-list fput i (sentence start-of-season end-of-season)
    set seasonal-evaporation-data lput new-year-list seasonal-evaporation-data
  ]

  ; Handle the last year by connecting it to the first year
  let last-year item (length evaporation-data - 1) evaporation-data
  let first-year item 0 evaporation-data
  let last-start-of-season sublist last-year cutoff-index 366
  ;let first-end-of-season sublist first-year 2 (cutoff-index + 1)
  let first-end-of-season sublist first-year 1 (cutoff-index)
  let last-year-list fput (length evaporation-data - 1) (sentence last-start-of-season first-end-of-season)
  set seasonal-evaporation-data lput last-year-list seasonal-evaporation-data

   ;Export the seasonal precipitation data to a CSV file
   ;csv:to-file "reconstructed evaporation.csv" seasonal-evaporation-data

  ; Update global evaporation-data with the newly constructed seasonal data
  set evaporation-data seasonal-evaporation-data
end

to initialize-time-indexes
  set-climate-random-years ;set list of the order of years of precipitation and evaporation data that will be used on model run
  ; Initialize the current year index with the first year in the random year indices list
  set current-year-index item 0 random-year-indices
  set current-day-index calculate-day-index current-month day
end



to update-precipitation ; Update the precipitation data for the current day
  ; Get the precipitation value from the current year and day index
  let current-year item current-year-index precipitation-data
  set daily-rainfall-amount (item current-day-index current-year) * climate-adjustment-factor
  ;set daily-rainfall-amount daily-rainfall-amount * climate-adjustment-factor

  ; Get the evaporation value from the current year and day index
  let current-year-evaporation item current-year-index evaporation-data
  set daily-evaporation-amount item current-day-index current-year-evaporation

  ; Update the monthly total rainfall amount
  set monthly-total-rainfall-amount monthly-total-rainfall-amount + daily-rainfall-amount

  ; Update the total precipitation (model running period total rainfall)
  set total-precipitation total-precipitation + daily-rainfall-amount

  ; Update the current year total precipitation
  set yearly-total-precipitation yearly-total-precipitation + daily-rainfall-amount

  ; Update the total and yearly evaporation
  set total-evaporation total-evaporation + daily-evaporation-amount
  set yearly-total-evaporation yearly-total-evaporation + daily-evaporation-amount


  ; Check if rainfall amount is greater than mean value of all Zetan catchment patches dry conditions (IaAMC1) and handle high rainfall
  if daily-rainfall-amount >= 7.28 [ handle-high-rainfall ]

  ; Update the current day index and check if it's the end of the year
  set current-day-index current-day-index + 1
  if current-day-index >= length current-year [
    set current-day-index 1 ; Reset to January 1st

    ; Move to the next year in the random year indices list
    let years-in-list length random-year-indices
    let current-year-position position current-year-index random-year-indices
    if current-year-position = (years-in-list - 1) [
      stop ; Stop the simulation when all years in the list have been processed
    ]
    set current-year-index item (current-year-position + 1) random-year-indices
  ]
end



; seting patch moisture state (dry = 1; wet = 2) based on excess water on patch;
; where excess water = daily-rainfall-amount - runoff-amount of runoff-here - daily-evaporation-amount
to update-antecedent-rainfall
  ask patches with [is-catchment-area] [
    ; Calculate runoff water amount if there are runoff agents on the patch
    let runoff-water ifelse-value (any? runoffs-here) [sum [runoff-amount] of runoffs-here] [0]
    ; Calculate the balance of daily rainfall, runoff, and evaporation, ensuring it's not negative
    let pre_balance daily-rainfall-amount - runoff-water - daily-evaporation-amount
    let balance max list 0 pre_balance  ; Ensures balance is not negative
    ; Remove the first item from patch-moisture-level and add the new balance at the end
    set patch-moisture-level fput balance (but-first patch-moisture-level)
    ; Check if the sum of patch-moisture-level is greater or equal to iaamc1
    ifelse sum patch-moisture-level >= iaamc1 [
      set patch-AMC 2  ; AMCII state, wet conditions
    ] [
      set patch-AMC 1  ; AMCI state, dry conditions
    ]
  ]
end


; This reporter calculates the index of the day in the year, given a month and a day.
; It is used to find the corresponding index in the precipitation data list.
to-report calculate-day-index [input-month input-day]
  ; Calculate the total number of days in all the previous months of the given month.
  ; It uses the calculate-days-in-month reporter to find the number of days in each month.
  let days-in-previous-months sum (map calculate-days-in-month (n-values (input-month - 1) [i -> i + 1]))

  ; The day index is the sum of the days in the previous months and the given day.
  report days-in-previous-months + input-day
end

; This reporter returns the current date as a string in the format "Month Day, Year".
; It uses the current-year-index, current-month, and day globals to determine the current date.
to-report report-current-date
  ; Calculate the current year by adding 1967 to the current-year-index.
  let current-UI-reported-year (current-year-index + 1952)

  ; Define the names of the months.
  let month-names ["January" "February" "March" "April" "May" "June" "July" "August" "September" "October" "November" "December"]

  ; Find the name of the current month using the current-month global.
  let current-month-name item (current-month - 1) month-names

  ; Concatenate the month name, day, and year to form the date string and report it.
  report (word current-month-name " " day ", " current-UI-reported-year)
end


to update-time ; Update the current time in the model and handle the end of the month and year
  ; Calculate the number of days in the current month only if it's the first day of the month
  if day = 1 [
    set days-in-month calculate-days-in-month current-month ; Pass the current-month as an argument
  ]

  ; Update day
  set day day + 1

  ; Check if it's the end of the month
  if day > (calculate-days-in-month current-month) [ ; Pass the current-month as an argument
    set monthly-total-rainfall-amount 0 ; Reset the monthly total rainfall amount for the new month
    set day 1
    set current-month current-month + 1
    if current-month > 12 [
      set current-month 1 ; resets current month back to January
      set year year + 1 ; advances model year by 1
      set yearly-total-precipitation 0 ; Reset the yearly total precipitation for the new year
      ;check-leap-year ; Check if the new year is a leap year
     ]
    if day = 31 and current-month = 12 [set yearly-total-precipitation 0] ;Reset the yearly total precipitation for the new grow season
    set months-passed months-passed + 1 ; Increment the number of months that have passed
    if months-passed >= 12 [
      ;ask vines [ set size 1 ] ; Reset the size of the vines to their starting size
      set months-passed 0 ; Reset the months-passed variable
    ]

  ]

end

to-report calculate-days-in-month [month-number] ; Calculate the number of days in the given month, considering leap years
  ifelse month-number = 2 [report 28]

  [
    ifelse member? month-number [4 6 9 11] [ report 30 ] ; April, June, September, November have 30 days
    [ report 31 ] ; All other months have 31 days
  ]
end

; track season based on UI tracked time
to update-current-season
  ifelse current-UI-month >= 11 or current-UI-month <= 3 [
    set current-season "winter"
  ] [
    ifelse current-UI-month >= 4 and current-UI-month <= 5 [
      set current-season "spring"
    ] [
      ifelse current-UI-month >= 6 and current-UI-month <= 8 [
        set current-season "summer"
      ] [
        set current-season "autumn"
      ]
    ]
  ]
end

to update-UI-tracked-time
  ; Update day
  set current-UI-day current-UI-day + 1

   if current-UI-day > (calculate-days-in-month current-UI-month) [ ; Pass the current UI tracked month as an argument

    set current-UI-day 1
    set current-UI-month current-UI-month + 1
    if current-UI-month > 12 [
      set current-UI-month 1 ; resets UI tracked month back to January
      set current-UI-year current-UI-year + 1 ; advances UI tracked year by 1
     ]
  ]

end

to handle-high-rainfall
  set high-rainfall-count high-rainfall-count + 1 ;keep score of how many floods occured

end


to-report average-yearly-precipitation ; Calculate and report the average yearly precipitation based on the total precipitation and the number of years passed
  ; Calculate the number of years that have passed
  let years-passed (year - 535)

  ; If no years have passed, avoid division by zero by reporting undefined or 0
  if years-passed = 0 [ report 0 ]

  ; Calculate and report the average yearly precipitation
  report (total-precipitation / years-passed)
end

to set-climate-adjustment-factor
; Determine the adjustment factor based on the climate conditions selected from the UI
    if climate-conditions = "same as present-day (87mm)" [
    set climate-adjustment-factor 1.00
  ]
  if climate-conditions = "10% wetter (96mm)" [
    set climate-adjustment-factor 1.10
  ]
  if climate-conditions = "25% wetter (109mm)" [
    set climate-adjustment-factor 1.25
  ]
  if climate-conditions = "10% drier (78mm)" [
    set climate-adjustment-factor 0.9
  ]
  if climate-conditions = "25% drier (65mm)" [
    set climate-adjustment-factor 0.75
  ]

end

; <<<<<<<<<<<<<<<<<<<<<<<<Farms Decision-Making procedures >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
to update-farms-failure-records
  ask farms [
    ; Calculate the wheat failure threshold based on given formula
    let wheat-failure-threshold ((wheat-season-labor + wheat-sow-amount) * maintenance-cost * dunam-to-patch-ratio * 0.5)
    let grape-failure-threshold 270  ; Static threshold for grapes

    ; Update wheat failure records
    ifelse annual-avg-wheat-yield < wheat-failure-threshold [
      set wheat-failure-records fput 1 butlast wheat-failure-records
    ]  [
      set wheat-failure-records fput 0 butlast wheat-failure-records
    ]

    ; Update grape failure records
    ifelse annual-avg-grape-yield < grape-failure-threshold [
      set grape-failure-records fput 1 butlast grape-failure-records
    ]  [
      set grape-failure-records fput 0 butlast grape-failure-records
    ]
  ]
end

; The farm-agriculture-expansion method decides whether each farm will expand its grape vineyards or wheat fields or take no action based on its annual average yield and historical failure records.
to farm-agriculture-expansion
  ask farms [
    ; If the farm's average annual grape yield is very high (> 400), expand vineyards immediately.
    if annual-avg-grape-yield > 400 [
      expand-vineyards
      ;print (word "finished  expand-vineyards for farm ID" farm-id " that annual-avg-grape-yield > 400, at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
      ;print ""
    ]
    ; If the average annual grape yield is moderate (between 270 and 400), calculate a decision threshold based on past grape failures.
    if annual-avg-grape-yield >= 270 and annual-avg-grape-yield <= 400 [
      let grape-decision-threshold calculate-decision-threshold grape-failure-records
      ;print (word "Farm ID " farm-id "grape-decision-threshold: " grape-decision-threshold)
      let decision-score random-float 1  ; Random value between 0 and 1
      ;print (word "Farm ID " farm-id "grape decision score: " decision-score)
      if decision-score > grape-decision-threshold [
        expand-vineyards
        ;print (word "finished  expand-vineyards for farm ID" farm-id " that annual-avg-grape-yield between 270 and 400 and scored above grape threshold, at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
        ;print ""
      ]
      if decision-score <= grape-decision-threshold [
        if annual-avg-wheat-yield > 10 [
        ; Calculate a decision threshold for wheat if grape expansion is not decided.
          let wheat-decision-threshold calculate-decision-threshold wheat-failure-records
          ;print (word "Farm ID " farm-id "wheat-decision-threshold: " wheat-decision-threshold)
          set decision-score random-float 1  ; New random value for wheat decision
          ;print (word "Farm ID " farm-id "wheat decision score: " decision-score)
          if decision-score > wheat-decision-threshold [
            expand-wheat-fields
            ;print (word "finished  expand-wheat-fields for farm ID" farm-id " that annual-avg-grape-yield between 270 and 400 and annual-avg-wheat-yield > 10 and scored above wheat threshold, at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
            ;print ""
          ]
        ]
      ]
    ]
    ; If the average annual grape yield is low (< 270), only consider expanding wheat fields.
    if annual-avg-grape-yield < 270 [
      if annual-avg-wheat-yield > 10 [
        let wheat-decision-threshold calculate-decision-threshold wheat-failure-records
        ;print (word "Farm ID " farm-id "wheat-decision-threshold: " wheat-decision-threshold)
        let decision-score random-float 1  ; Random value between 0 and 1
        ;print (word "Farm ID " farm-id "wheat decision score: " decision-score)
        if decision-score > wheat-decision-threshold [
          expand-wheat-fields
          ;print (word "finished  expand-wheat-fields for farm ID" farm-id " that annual-avg-grape-yield less then 270  and annual-avg-wheat-yield > 10 and scored above wheat threshold, at time stamp: " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")
          ;print ""
        ]
      ]
    ]
  ]
end

; The calculate-decision-threshold function calculates a threshold for making decisions on expansion based on the historical failure records.
to-report calculate-decision-threshold [failure-records]
  report 0.5 + (sum (map calculate-failure-contribution failure-records))
end

; The calculate-failure-contribution function adds a contribution to the decision threshold based on each failure record.
to-report calculate-failure-contribution [failure]
  report ifelse-value (failure = 1) [0.5 / farms-memory-length] [0]
end




;<<<<<<<<<<<<<<<<<<<<<<<< Procedures controlling display on map >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
;;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; Procedure to display geographic features on the map.
to display-geographic-features

  ; Highlight catchment area borders in white (pcolor 45).
  ask patches with [is-catchment-border] [set pcolor 45]

  ; Color city area and borders in grey (pcolor 25).
  ask patches with [is-city-border or is-city-area] [set pcolor 25]

  ; Color tributary patches in blue (pcolor 96), excluding those within the city area.
  ask patches with [is-tributary and not is-city-area ] [set pcolor 96]

  ; Color main river patches in white.
  ask patches with [is-main-river] [set pcolor white]
end

; Procedure to color the topographic map based on elevation.
to color-topographic-map
  ; Calculate the minimum and maximum elevation values across all patches.
  let min-elevation min [patch-elevation] of patches
  let max-elevation max [patch-elevation] of patches
  let elevation-range (max-elevation - min-elevation)
  let base-color 35  ; Set a base brown color for elevation coloring.

  ; Assign a color to each patch based on its relative elevation.
  ask patches [
    let relative-elevation ((patch-elevation - min-elevation) / elevation-range)
    let color-adjustment (relative-elevation * 9 - 4.5)  ; Adjust color based on elevation.
    set pcolor base-color + color-adjustment ; Apply color adjustment to base color.
  ]
end

;; making the patches that are runoff harvest areas (Tulilat El-Anab) visible
to set-mounds-terrain-scheme
  ask patches with [patch-runoff-cluster-terrace-id != -1 and not any? mounds-here] [
    sprout-mounds 1 [
      set shape "mounds"
      set size 0.7
    ]
  ]
end

to color-basins
  ; Create a list to store unique basin-ids
  let basin-ids remove-duplicates [patch-sub-basin-id] of patches

  ; Generate a list of colors
  let available-colors [8 18 28 48 58 78 118 128 138]

  ; Initialize an empty list for basin-color-map
  let basin-color-map []

  ; Iterate over the basin-ids and create a list of basin-id and color pairs
  foreach basin-ids [
    basin-id ->
    let color-index random length available-colors
    let temp-color item color-index available-colors
    set basin-color-map lput (list basin-id temp-color) basin-color-map
  ]

  ; Color each patch based on its basin-id
  ask patches with [is-catchment-area] [
    let my-basin-id patch-sub-basin-id
    let my-color-entry first filter [entry -> first entry = my-basin-id] basin-color-map
    if (my-color-entry != []) [
      set pcolor item 1 my-color-entry  ; Set color from the color map entry
    ]
  ]
  display-geographic-features
end



to update-farms-icons
  ask farms [
    let current-farm-id farm-id
  let capital (wheat-storage - wheat-storage * farm-wheat-reserve)
  let cost-to-expand dunam-to-patch-ratio * maintenance-cost * (build-grapevine-terrace + prepare-harvest-runoff-field * runoff-ratio + vineyard-planting + vineyard-establishment-phase + vineyard-season-labor)

    ifelse (capital >= cost-to-expand) [
      set color green
    ] [
  ; Check if the farm does not have enough capital to expand but has at least one active grapevine terrace
    ifelse any? terraces with [crop-type = "grapevine" and owned-by-farm = current-farm-id and is-active = true] [
      set color yellow
    ] [
    set color red
      ]
    ]
  ]

end


;; set visualization of terrace status
to set-terraces-icons
  ask terraces [
    if not is-active [
      set shape "empty-terrace"
    ]
    if is-active and crop-type = "wheat" [
      set shape "wheat"
    ]
    if is-active and crop-type = "grapevine" [
      set shape "vineyard"
    ]
  ]

end

;;<<<<<<<<<<<<<<<<< Seting patches attributes according to intersection with cerain shape file data >>>>>>>>>>>>>>>>>>>>>>>>>>>>
;;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


to assign-topographic-data-to-patches
  ; Load the elevation shapefile dataset.
  let shp-dataset gis:load-dataset "data/Joined Elev Slopes and Flows/Joined Elev Slopes and Flows.shp"

  ; Extract the list of features (points) from the shapefile.
  let features gis:feature-list-of shp-dataset

  ; Iterate over each feature (point) in the shapefile.
  foreach features [
    this-feature ->
   ; Obtain the list of vertices for each feature (typically one for point features).
    let vertices gis:vertex-lists-of this-feature

    ; Proceed only if there is at least one vertex.
    if not empty? first vertices [
      ; Retrieve the coordinates of the first vertex.
      let vertex first first vertices
      let location gis:location-of vertex
      let patch-xcor item 0 location
      let patch-ycor item 1 location

     ; Extract the elevation value from the feature's property.
      let elevation-value gis:property-value this-feature "Elevation"
      let slope-deg-value gis:property-value this-feature "SlopeDEG"
      let slope-pr-value gis:property-value this-feature "SlopePR"
      let aspec-value gis:property-value this-feature "Aspect"
      let flow-direction-value gis:property-value this-feature "Direction"
      let flow-accumulation-value gis:property-value this-feature "Accum"

      ; Assign the elevation value to the corresponding patch within the world bounds.
      if (patch-xcor >= min-pxcor and patch-xcor <= max-pxcor) and (patch-ycor >= min-pycor and patch-ycor <= max-pycor) [
        ask patch patch-xcor patch-ycor [
          set patch-elevation elevation-value
          set patch-slope-deg slope-deg-value
          set patch-slope-pr slope-pr-value
          set patch-aspect aspec-value
          set patch-flow-direction flow-direction-value
          set patch-flow-accumulation flow-accumulation-value
        ]
      ]
    ]
  ]
end


; Procedure to calculate and assign patch facing directions based on their aspect values.
to assign-patch-face

  ; Iterate over all patches in the world.
  ask patches [
    ; Determine the facing direction based on the aspect value and assign the corresponding compass direction.
    ifelse (patch-aspect >= 0 and patch-aspect < 22.5) or (patch-aspect >= 337.5 and patch-aspect <= 360) [
      set patch-face "North"
    ] [
      ifelse (patch-aspect >= 22.5 and patch-aspect < 67.5) [
        set patch-face "North East"
      ] [
        ifelse (patch-aspect >= 67.5 and patch-aspect < 112.5) [
          set patch-face "East"
        ] [
          ifelse (patch-aspect >= 112.5 and patch-aspect < 157.5) [
            set patch-face "South East"
          ] [
            ifelse (patch-aspect >= 157.5 and patch-aspect < 202.5) [
              set patch-face "South"
            ] [
              ifelse (patch-aspect >= 202.5 and patch-aspect < 247.5) [
                set patch-face "South West"
              ] [
                ifelse (patch-aspect >= 247.5 and patch-aspect < 292.5) [
                  set patch-face "West"
                ] [
                  set patch-face "North West"  ; This covers 292.5 to 337.5 degrees.
                ]
              ]
            ]
          ]
        ]
      ]
    ]
  ]
end


; Procedure to determine if a patch intersects with Shivta area stream network of tributaries type (shapefile).
to set-stream-order
  ; Initialize all patches with a default stream-order of -1.
  ask patches [
    set stream-order -1
  ]

  ; Load the shapefile containing stream network data.
  let stream-dataset gis:load-dataset "data/Zetan Streams Network/Zetan Stream Network Cliped.shp"

  ; Get a list of all features (lines) in the shapefile.
  let features gis:feature-list-of stream-dataset

  ; Iterate over each feature in the shapefile.
  foreach features [
    this-feature ->
    ; Get the stream order from the feature's 'grid_code' property.
    let order-value gis:property-value this-feature "grid_code"

    ; Assign the stream order to intersecting patches.
    ask patches with [gis:intersects? this-feature self] [
      set stream-order order-value
    ]
  ]
end

to assign-streams-IDs
  print date-and-time
  print "Zetan streams (IDs) catchment polygons file data loading started"
  ; Load the polygon shapefile
  let streams-polygon-dataset gis:load-dataset "data/Zetan Stream Network IDs/Zetan Streams IDs Catchment Polygons.shp"


  ; Get a list of all features (polygons) in the shapefile
  let features gis:feature-list-of streams-polygon-dataset


  ; Iterate over each feature in the shapefile
  foreach features [
    this-feature ->


    ; Get the stream ID from the feature's 'arcid' property
    let arcid-value gis:property-value this-feature "arcid"


    ; Assign the stream ID to patches within this polygon feature
    ask patches with [gis:intersects? this-feature self] [
      set stream-id arcid-value
    ]

  ]

  ; Resets stream-id if any patch not part of stream network got a stream-id other than -1
  ask patches with [stream-order = -1] [set stream-id -1]

  print date-and-time
  print "Zetan streams (IDs) catchment polygons file data loading completed"
end



; Procedure to determine if a patch intersects with the catchment boundary polygon.
to set-catchment-area
  ; Load the catchment boundary dataset.
  let catchment-dataset gis:load-dataset "data/Zetan basin data/Zitan_basin.shp"

  ; Iterate over each patch and set is-catchment-area based on intersection.
  ask patches [
    ; Initially assume the patch is not in the catchment area.
    set is-catchment-area false

    ; Check if the current patch intersects with the catchment boundary.
    if gis:intersects? catchment-dataset self [
      set is-catchment-area true
    ]
  ]
end

; Procedure to determine if a patch intersects with the city boundary.
to set-city-area
  ; Load the city boundary dataset.
  let city-dataset gis:load-dataset "data/Shivta city data/Shivta_city.shp"

  ; Iterate over each patch and set is-city-area based on intersection.
  ask patches [
    ; Initially assume the patch is not in the city area.
    set is-city-area false

    ; Check if the current patch intersects with the city boundary.
    if gis:intersects? city-dataset self [
      set is-city-area true
    ]
  ]
end

; Procedure to determine city border patches and collect their coordinates.
to set-city-border
  ; Initialize the list as empty.
  set city-border-coordinates []

  ; Assume no patches are on the city border initially and find border patches.
  ask patches [
    set is-city-border false
    if is-city-area [
      if any? neighbors with [not is-city-area] [
        set is-city-border true
        ; Add the patch coordinates to the list.
        ;Set city-border-coordinates lput (list pxcor pycor) city-border-coordinates
      ]
    ]
  ]

end


; Procedure to determine catchment border patches and collect their coordinates.
to set-catchment-border
  ; Initialize the list as empty.
  set catchment-border-coordinates []

  ; Assume no patches are on the catchment border initially and find border patches.
  ask patches [
    set is-catchment-border false
    if is-catchment-area [
      if any? neighbors with [not is-catchment-area] [
        set is-catchment-border true
        ; Add the patch coordinates to the list.
        ;Set catchment-border-coordinates lput (list pxcor pycor) catchment-border-coordinates
      ]
    ]
  ]

end

; Procedure to assign soil group type data to patches based on a polygon shapefile.
to set-soil-data
  ; Load the soil group type shapefile.
  let soil-dataset gis:load-dataset "data/SoilGroupClipped/SoilClip.shp"

  ; Get a list of all features in the shapefile.
  let features gis:feature-list-of soil-dataset

  ; Iterate over each feature.
  foreach features [
    this-feature ->
    ; Get the soil type from the feature's 'HK_Type' property.
    let soil-type gis:property-value this-feature "HK_Type"

    ; Assign the soil type to intersecting patches.
    ask patches with [gis:intersects? this-feature self] [
      set patch-soil-index soil-type
    ]
  ]
end

; Procedure to analyze patch soil index and assign appropriate runoff coefficients.
to set-runoff-coefficients
  ; Iterate over all patches and assign runoff coefficients based on soil type.
  ask patches [
    ; Check if the soil index is valid.
    if (is-string? patch-soil-index) and not empty? patch-soil-index [
      ; Assign runoff coefficients based on soil group initial.
      ifelse first patch-soil-index = "S" [
        set patch-runoff-coefficient 0.20
      ] [
        ifelse first patch-soil-index = "R" [
          set patch-runoff-coefficient 0.80
        ] [
          ifelse first patch-soil-index = "X" [
            set patch-runoff-coefficient 0.40
          ] [
          ]
        ]
      ]
    ]
  ]
end


; Procedure to assign sub-basin ID values to patches from a polygon shapefile.
to set-sub-basin-id

  ; Load the sub-basin shapefile.
  let subBasin-dataset gis:load-dataset "data/Zetan Streams SubBasins/Zetan Streams Sub Basins Cliped.shp"

  ; Get a list of all features in the shapefile.
  let features gis:feature-list-of subBasin-dataset

  ; Iterate over each feature.
  foreach features [
    this-feature ->
    ; Get the sub-basin ID from the feature's property.
    let subBasin-id gis:property-value this-feature "Id"

    ; Assign the sub-basin ID to intersecting patches.
    ask patches with [gis:intersects? this-feature self] [
      set patch-sub-basin-id subBasin-id
    ]
  ]
  print date-and-time
  print "Zetan sub-basins (IDs) polygons file data loading completed"
end

;;<<<<<<<<<<<<<<<<< Exporting / Reading data procedures  (to/from text files) >>>>>>>>>>>>>>>>>>>>>>>>>>>>
;;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; Procedure to export static patch values to a text file for faster setup in future runs.
to export-static-world-patches-values
  ; Check if the file exists and delete it to prevent appending to old data.
  if file-exists? "AGENTS 40K model patches dataset.txt" [
    file-delete "AGENTS 40K model patches dataset.txt"
  ]

  ; Open the file to write patch attributes.
  file-open "AGENTS 40K model patches dataset.txt"

  ; Write attributes of each patch to the file.
  ask patches [
    let patch-attributes (list pxcor pycor patch-elevation patch-slope-deg patch-slope-pr patch-aspect patch-face
                          is-catchment-area is-catchment-border is-city-area is-city-border
                           stream-order patch-soil-index patch-runoff-coefficient patch-flow-direction patch-flow-accumulation
                            patch-sub-basin-id stream-id)

    file-write patch-attributes
  ]
  ; Close the file after writing all data.
  file-close
end

; Procedure to load patch values from an external text file.
to load-patch-values
  ; Open the file containing the patch values
  print "loading data to patches from external file started"
  ; Announce the start of data loading and record the time.
  print date-and-time

  ; Open the file containing previously saved patch attributes.
  file-open "AGENTS 40K model patches dataset.txt"

  ; Read each line from the file and assign values to patch attributes.
  while [not file-at-end?] [
    let patch-attrs file-read
    if is-list? patch-attrs and length patch-attrs = 18 [
      ; Extract patch coordinates and attribute values.
      let patch-pxcor item 0 patch-attrs
      let patch-pycor item 1 patch-attrs
      let pelevation item 2 patch-attrs
      let pslope-DEG item 3 patch-attrs
      let pslope-PR item 4 patch-attrs
      let paspect item 5 patch-attrs
      let pface item 6 patch-attrs
      let catchment-area item 7 patch-attrs
      let catchment-border item 8 patch-attrs
      let city-area item 9 patch-attrs
      let city-border item 10 patch-attrs
      let pstrmordr item 11 patch-attrs
      let psoil item 12 patch-attrs
      let prunoff item 13 patch-attrs
      let pflowdir item 14 patch-attrs
      let pflowacc item 15 patch-attrs
      let psubbasinid item 16 patch-attrs
      let psreamid item 17 patch-attrs

      ; Update each patch with extracted values.
      ask patch patch-pxcor patch-pycor [
        ; Set the extracted values to respective patch variables.
        set patch-elevation pelevation
        set patch-slope-deg pslope-DEG
        set patch-slope-pr pslope-PR
        set patch-aspect paspect
        set patch-face pface
        set is-catchment-area catchment-area
        set is-catchment-border catchment-border
        set is-city-area city-area
        set is-city-border city-border
        set stream-order pstrmordr
        set patch-soil-index psoil
        set patch-runoff-coefficient prunoff
        set patch-flow-direction pflowdir
        set patch-flow-accumulation pflowacc
        set patch-sub-basin-id psubbasinid
        set stream-id psreamid
      ]
    ]
  ]

  ; Close the file after completing the data loading.
  file-close
  print "loading data to patches from external file completed"

end


;;<<<<<<<<<<<<<<<<<<<<<<<<<<<< UI Statistics & Plots Procedures >>>>>>>>>>>>>>>>>>>>>>>>>

to report-end-of-run-grape-yield
  ; Create an empty list to hold pairs of farm ID and total grape yield
  let farm-yield-pairs []

  ; Iterate through each farm
  ask farms [
    let temp-farm-id farm-id  ; Store the farm ID in a temporary variable
    let temp-total-grape-yield 0  ; Initialize the total grape yield for the farm

    ; Calculate the total grape yield for the farm
    ask terraces with [owned-by-farm = temp-farm-id and crop-type = "grapevine"] [
      let temp-terrace-yield sum annual-terrace-yield-list  ; Total yield for this terrace
      set temp-total-grape-yield temp-total-grape-yield + temp-terrace-yield
    ]

    ; Add the pair (farm ID, total grape yield) to the list
    set farm-yield-pairs lput (list temp-farm-id temp-total-grape-yield) farm-yield-pairs
  ]

  ; Update the plot with the calculated yields
  set-current-plot "Total Grape Yield by Farm"
  clear-plot
  set-current-plot-pen "default"
  set-plot-x-range 0 number-of-farms

  ; Iterate through each pair and plot the yield
  foreach farm-yield-pairs [
    farm-yield-pair ->
    if length farm-yield-pair = 2 and item 0 farm-yield-pair >= 0 [ ; Check if the pair is valid
      plotxy item 0 farm-yield-pair item 1 farm-yield-pair
    ]
  ]
  ;print (word "Farm:total grape yield pairs: " farm-yield-pairs)
end

to report-end-of-run-wheat-yield
  ; Create an empty list to hold pairs of farm ID and total wheat yield
  let farm-yield-pairs []

  ; Iterate through each farm
  ask farms [
    let temp-farm-id farm-id  ; Store the farm ID in a temporary variable
    let temp-total-wheat-yield 0  ; Initialize the total wheat yield for the farm

    ; Calculate the total wheat yield for the farm
    ask terraces with [owned-by-farm = temp-farm-id and crop-type = "wheat"] [
      let temp-terrace-yield sum annual-terrace-yield-list  ; Total yield for this terrace
      set temp-total-wheat-yield temp-total-wheat-yield + temp-terrace-yield
    ]

    ; Add the pair (farm ID, total wheat yield) to the list
    set farm-yield-pairs lput (list temp-farm-id temp-total-wheat-yield) farm-yield-pairs
  ]

  ; Update the plot with the calculated yields
  set-current-plot "Total Wheat Yield by Farm"
  clear-plot
  set-current-plot-pen "default"
  set-plot-x-range 0 number-of-farms

  ; Iterate through each pair and plot the yield
  foreach farm-yield-pairs [
    farm-yield-pair ->
    if length farm-yield-pair = 2 and item 0 farm-yield-pair >= 0 [ ; Check if the pair is valid
      plotxy item 0 farm-yield-pair item 1 farm-yield-pair
    ]
  ]
  ;print (word "Farm:total wheat yield pairs: " farm-yield-pairs)
end

; Procedure to update the wine production plot based on the agrisys-annual-wine-production-list
to update-wine-production-plot
  ; Prepare the plot for new data
  set-current-plot "Annual Wine Production"
  clear-plot
  set-plot-x-range 0 model-run-period

  ; Iterate through each year and wine production pair in the list
  foreach agrisys-annual-wine-production-list [
    pair ->
    let plot-year item 0 pair
    let plot-wine-production item 1 pair

    ; Plot the year and wine production
    plotxy plot-year plot-wine-production
  ]
end



; Procedure to print the total yield for each farm by crop type, across all years
to print-farm-total-yields
  ; Print report title
  print (word "Farms total yields for model run over " model-run-period " years (runoff ratio 1:" runoff-ratio "): ")
  print (word "printout format: total yield in kg; [number of terraces]; (avg/year); {efficiency}")
  print ("-------------------------------------------------------------------------")

  ; Determine the number of farms
  let num-farms count farms

  ; Initialize a counter for the farm ID
  let current-farm-id 0

  ; Iterate over the farm IDs
  while [current-farm-id < num-farms] [
    ; Find the farm with the current farm ID
    let current-farm one-of farms with [farm-id = current-farm-id]

    ; Calculate average efficiencies for wheat and grapevine terraces
    let wheat-efficiencies [annual-terrace-yield-efficiency] of terraces with [crop-type = "wheat" and owned-by-farm = current-farm-id]
    let grapevine-efficiencies [annual-terrace-yield-efficiency] of terraces with [crop-type = "grapevine" and owned-by-farm = current-farm-id]

    let avg-wheat-efficiency ifelse-value (not empty? wheat-efficiencies) [mean reduce sentence wheat-efficiencies] [0]
    let avg-grapevine-efficiency ifelse-value (not empty? grapevine-efficiencies) [mean reduce sentence grapevine-efficiencies] [0]


    ; Access and print the total yields from the farm attributes
    ask current-farm [
      let total-grapevine-yield precision (total-grape-yield-kg) 0 ; Convert to tons
      let total-wheat-yield precision (total-wheat-yield-kg) 0  ; Convert to tons

      ; Print report data
      ;print (word "Farm ID: " farm-id ", Grapes: " total-grapevine-yield " kg (AVG: " (word precision (total-grapevine-yield / model-run-period) 1 " kg/year); Wheat: " total-wheat-yield " kg (AVG: " (word precision (total-wheat-yield / model-run-period) 1 " kg/year) .")))
      if (total-grapevine-yield > 0 and total-wheat-yield > 0) [
        print (word farm-id ",grape data," total-grapevine-yield "," total-grape-terraces "," (word precision (total-grapevine-yield / model-run-period) 1 "," precision (avg-grapevine-efficiency) 3 ",wheat data," total-wheat-yield "," total-wheat-terraces "," (word precision (total-wheat-yield / model-run-period) 1 ","precision (avg-wheat-efficiency) 3 )))
      ]
      if (total-grapevine-yield > 0 and total-wheat-yield = 0) [
        print (word  farm-id ",grapes data," total-grapevine-yield "," total-grape-terraces "," (word precision (total-grapevine-yield / model-run-period) 1 "," precision (avg-grapevine-efficiency) 3 ))
        ]
        if (total-wheat-yield > 0 and total-grapevine-yield = 0) [
          print (word  farm-id ",wheat data,"total-wheat-yield "," total-wheat-terraces "," (word precision (total-wheat-yield / model-run-period) 1 "," precision (avg-wheat-efficiency) 3))
      ]
]

    ; Increment the farm ID counter
    set current-farm-id current-farm-id + 1
  ]
end





;;<<<<< External Procedures for Debugging and Calculations >>>>>>>>>>>>>>>>>>
;;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

to print-inactive-farms
  let inactive-farms (list)  ; Initialize an empty list to track farms with all inactive terraces

  ask farms [
    if not any? terraces with [is-active] [
      set inactive-farms lput farm-id inactive-farms  ; Add the farm-id to the list if all terraces are inactive
    ]
  ]

  ifelse empty? inactive-farms [
    print "No such farms found with all terraces inactive."
  ] [
    foreach inactive-farms [
      temp-farm-id ->
      print (word "Farm ID " farm-id " has all inactive terraces.")
    ]
  ]
end


to print-active-grapevine-terraces-info
  ; Collect all grapevine terraces and print their coordinates
  let grapevine-terraces terraces with [crop-type = "grapevine"]
  let grapevine-info (list count grapevine-terraces)
  ask grapevine-terraces [
    set grapevine-info fput (list pxcor pycor) grapevine-info
  ]
  print grapevine-info
  ; Collect all wheat terraces and print their coordinates
  let wheat-terraces terraces with [crop-type = "wheat"]
  let wheat-info (list count wheat-terraces)
  ask wheat-terraces [
    set wheat-info fput (list pxcor pycor) wheat-info
  ]
  print wheat-info

  ; Collect all patches with coverage-status "clear with mounds" and print their coordinates
  let clear-mounds-patches patches with [coverage-status = "clear with mounds"]
  let mounds-info (list count clear-mounds-patches)
  ask clear-mounds-patches [
    set mounds-info fput (list pxcor pycor) mounds-info
  ]
  print  mounds-info
end

to calculate-run-time
  ;let time-passed timer
  ;let total-minutes floor (timer / 60)
  ;let total-seconds floor (remainder timer 60)
  print (word "Run time is " (floor (timer / 60)) "m, " (floor (remainder timer 60)) " s.")

end
@#$#@#$#@
GRAPHICS-WINDOW
275
107
1449
784
-1
-1
4.4
1
10
1
1
1
0
0
0
1
0
264
0
151
0
0
1
ticks
30.0

BUTTON
79
70
142
103
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
421
59
1284
88
AGENTS: Agriculture Grape (yield) Evaluation (using) NetLogo (based) Technology Simulation
20
105.0
1

BUTTON
18
70
73
103
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
16
336
188
369
model-run-period
model-run-period
1
50
10.0
1
1
years
HORIZONTAL

SLIDER
15
380
187
413
runoff-ratio
runoff-ratio
1
30
27.0
1
1
NIL
HORIZONTAL

TEXTBOX
19
22
275
62
Quick-Setup must be run after any change made to model sliders\\switches
14
14.0
1

MONITOR
1213
119
1270
180
Day
current-UI-day
0
1
15

MONITOR
1275
118
1332
179
Month
current-UI-month
0
1
15

MONITOR
1338
118
1423
179
Year A.D.
current-UI-year
0
1
15

MONITOR
17
512
264
593
Avg. Yearly Rainfall (mm)
average-yearly-precipitation
0
1
20

MONITOR
19
740
265
821
Current CSV file Date
report-current-date
0
1
20

PLOT
1447
152
1931
302
Season-to-Season precipitations (mm)
NIL
NIL
0.0
1.0
0.0
1.0
true
false
"set-plot-x-range 0 model-run-period\nset-plot-x-range 0 model-run-period" "set-plot-x-range 0 model-run-period"
PENS
"pen-1" 1.0 0 -2674135 true "" "if (current-month = 1 and day = 1) [plot drought-threshold]"
"pen-2" 1.0 0 -13840069 true "" "if (current-month = 1 and day = 1) [plot 87.6]"
"pen-3" 1.0 1 -13345367 true "" "if (current-month = 12 and day = 31) [plot yearly-total-precipitation]"

MONITOR
145
603
260
664
W. Fields
count terraces with [crop-type = \"wheat\"]
0
1
15

SLIDER
17
425
189
458
number-of-farms
number-of-farms
1
50
17.0
1
1
NIL
HORIZONTAL

MONITOR
1129
120
1207
181
Season
current-season
17
1
15

MONITOR
985
120
1125
181
Daily rainfall (mm)
daily-rainfall-amount
2
1
15

BUTTON
154
70
260
103
Set Terraces Size 1
ask terraces [ set size 1 ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1449
467
1931
617
Total Grape Yield by Farm
Farm ID
Yield (kg)
0.0
1.0
0.0
1.0
true
false
"set-plot-x-range 0 number-of-farms\nset-plot-x-range 0 number-of-farms" "set-plot-x-range 0 number-of-farms"
PENS
"default" 1.0 1 -13840069 true "" ""

PLOT
1449
624
1931
774
Total Wheat Yield by Farm
Farm ID
Yield (kg)
0.0
1.0
0.0
1.0
true
false
"set-plot-x-range 0 number-of-farms\nset-plot-x-range 0 number-of-farms" "set-plot-x-range 0 number-of-farms"
PENS
"default" 1.0 1 -6459832 true "" ""

MONITOR
18
602
138
663
G. Terraces
count terraces with [crop-type = \"grapevine\"]
0
1
15

SLIDER
17
469
190
502
farm-wheat-reserve
farm-wheat-reserve
0
0.5
0.1
0.05
1
NIL
HORIZONTAL

SLIDER
15
297
188
330
consecutive-droughts-period
consecutive-droughts-period
1
10
3.0
1
1
NIL
HORIZONTAL

SLIDER
15
255
187
288
drought-threshold
drought-threshold
50
70
66.0
1
1
mm
HORIZONTAL

SWITCH
17
211
188
244
consecutive-droughts
consecutive-droughts
0
1
-1000

PLOT
1447
310
1931
460
Annual Wine Production
NIL
Wine (liters)
0.0
1.0
0.0
1.0
true
false
"set-plot-x-range 0 model-run-period" "set-plot-x-range 0 model-run-period"
PENS
"default" 1.0 1 -2674135 true "" "if (ticks > 0 and current-month = 1 and day = 1) [update-wine-production-plot]"

MONITOR
17
671
260
732
% abandonment  vineyards
;count terraces with [crop-type = \"grapevine\" and not is-active] / count terraces with [crop-type = \"grapevine\"]\n;100 * (1 - (count terraces with [crop-type = \"grapevine\" and not is-active]) / (count terraces with [crop-type = \"grapevine\"]))\n100 *  (count terraces with [crop-type = \"grapevine\" and not is-active]) / (count terraces with [crop-type = \"grapevine\"])
1
1
15

MONITOR
1445
39
1525
100
Not active
count farms with [color = red]
0
1
15

MONITOR
1531
40
1657
101
active producers
count farms with [color = yellow]
0
1
15

MONITOR
1661
39
1788
100
active expanders
count farms with [color = green]
0
1
15

TEXTBOX
1533
15
1743
35
Farms Agriculture Status
15
0.0
1

TEXTBOX
1449
132
1786
150
Shivta average annual rainfall - Green; Drought threshold - Red
12
0.0
1

SLIDER
17
114
189
147
farms-memory-length
farms-memory-length
3
7
5.0
1
1
NIL
HORIZONTAL

CHOOSER
18
157
189
202
climate-conditions
climate-conditions
"same as present-day (87mm)" "10% wetter (96mm)" "25% wetter (109mm)" "10% drier (78mm)" "25% drier (65mm)"
0

@#$#@#$#@
## WHAT IS IT?

The AGENTS V1.0.0 model simulates the socio-ecological dynamics of agricultural systems in the Byzantine Negev Highlands, focusing on the interplay between environmental, social, and economic factors. It explores how historical agricultural practices, such as runoff harvesting and terrace farming, influenced settlement sustainability and crop production, particularly for cash crops like grapevines.

## HOW IT WORKS

Agents represent farms, which make decisions based on resource availability, environmental conditions, and economic profitability. The model integrates hydrology (runoff harvesting), topography (DEM-derived slope and patch sizes), climate (precipitation scenarios), and social behavior (resource allocation and terrace maintenance). The terraces and fields yield crops based on climate inputs, soil properties (curve numbers), and farm upkeep decisions. Inactive terraces or fields produce no yield if maintenance costs cannot be met.

## HOW TO USE IT

A series of YouTube videos explaining how to use the model are available. These videos were created for the MA thesis research that used this model (V1.0.0), which focused on Byzantine Shivta viticulture.

**Part 1:** Introduction to model user interface, covering main controls, sliders, monitors, and the central simulation world map. [Introduction to AGENTS model](https://youtu.be/fzhIniZosUg).

**Part 2:** Detailed explanation of the setup controllers, covering adjustments for initial conditions, climate scenarios, and simulation duration. [AGENTS model UI components](https://youtu.be/rmeFqwEObTY).

**Part 3:** Overview of the setup routine, from initializing global variables to creating farms and applying climate and hydrology data. [AGENTS model SETUP routine](https://youtu.be/-qb24YyxxMo).

**Part 4:** Breakdown of the go routine, describing the sequence of daily functions, seasonal transitions, and end-of-model run procedures. [AGENTS model GO routine](https://youtu.be/d87EIC10YcE).

## THINGS TO NOTICE


* Observe how varying precipitation levels or runoff ratios impact crop yields and the number of active terraces.
* Pay attention to economic profitability trends across farms and their ability to maintain terraces or fields.
* Notice the relationship between sub-watershed characteristics and runoff harvesting efficiency.


## THINGS TO TRY

* Experiment with different climate scenarios to see how droughts or wetter years influence terrace farming.
* Change the runoff-ratio to test the effectiveness of different runoff harvesting strategies.
* Alter the maintenance costs or raw-to-processed ratios to explore their effect on farm profitability.

## EXTENDING THE MODEL

* Add additional crops or agricultural systems to diversify agent decision-making.
* Incorporate alternative water management systems or technologies beyond traditional runoff harvesting.
* Model social interactions between farms, such as resource-sharing or competition for land and water.

## NETLOGO FEATURES

* The model uses patch attributes to represent soil, slope, and runoff contributions, with GIS-derived data input during setup.
* Workarounds were implemented to dynamically assign curve numbers (CN2) and patch-based water distributions based on soil groups.
* BehaviorSpace integration allows testing of different precipitation scenarios, including randomized drought periods.


## CREDITS AND REFERENCES

To cite this model, use:

Garty, B., Bar-Oz, G., Gambash, G., and Levy, S. T. (2024). Agriculture.Grape.yield.Evaluation.using.NetLogo.based.Technology (AGENTS) simulation: A NetLogo agent-based model developed to assess viticulture efficiency in Byzantine Shivta. Systems Learning & Development Lab, University of Haifa, Israel.




@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

drop
false
0
Circle -7500403 true true 73 133 152
Polygon -7500403 true true 219 181 205 152 185 120 174 95 163 64 156 37 149 7 147 166
Polygon -7500403 true true 79 182 95 152 115 120 126 95 137 64 144 37 150 6 154 165

empty-terrace
false
0
Circle -1 true false 2 2 297
Polygon -2674135 true false 270 75 225 30 30 225 75 270
Polygon -2674135 true false 30 75 75 30 270 225 225 270

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

mounds
true
0
Line -6459832 false 60 30 240 30
Line -6459832 false 15 90 285 90
Line -6459832 false 0 150 300 150
Line -6459832 false 15 210 285 210
Line -6459832 false 60 270 240 270
Circle -6459832 false false 45 45 30
Circle -6459832 false false 90 45 30
Circle -6459832 false false 135 45 30
Circle -6459832 false false 180 45 30
Circle -6459832 false false 225 45 30
Circle -6459832 false false 240 105 30
Circle -6459832 false false 195 105 30
Circle -6459832 false false 150 105 30
Circle -6459832 false false 105 105 30
Circle -6459832 false false 60 105 30
Circle -6459832 false false 15 105 30
Circle -6459832 false false 15 165 30
Circle -6459832 false false 60 165 30
Circle -6459832 false false 105 165 30
Circle -6459832 false false 150 165 30
Circle -6459832 false false 195 165 30
Circle -6459832 false false 240 165 30
Circle -6459832 false false 45 225 30
Circle -6459832 false false 90 225 30
Circle -6459832 false false 135 225 30
Circle -6459832 false false 180 225 30
Circle -6459832 false false 225 225 30
Line -6459832 false 66 274 234 274
Line -6459832 false 15 206 285 206
Line -6459832 false 0 147 300 147
Line -6459832 false 15 86 285 86
Line -6459832 false 68 25 233 24

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tile water
false
0
Rectangle -7500403 true true -1 0 299 300
Polygon -1 true false 105 259 180 290 212 299 168 271 103 255 32 221 1 216 35 234
Polygon -1 true false 300 161 248 127 195 107 245 141 300 167
Polygon -1 true false 0 157 45 181 79 194 45 166 0 151
Polygon -1 true false 179 42 105 12 60 0 120 30 180 45 254 77 299 93 254 63
Polygon -1 true false 99 91 50 71 0 57 51 81 165 135
Polygon -1 true false 194 224 258 254 295 261 211 221 144 199

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

vineyard
true
0
Circle -1 true false 0 0 300
Polygon -14835848 true false 120 45 180 45 150 120 135 105 120 45
Circle -8630108 true false 120 90 60
Circle -8630108 true false 180 90 60
Circle -8630108 true false 60 90 60
Circle -8630108 true false 90 135 60
Circle -8630108 true false 150 135 60
Circle -8630108 true false 120 180 60

vineyard-green
true
0
Polygon -6459832 true false 120 45 180 45 150 120 135 105 120 45
Circle -13840069 true false 120 90 60
Circle -13840069 true false 180 90 60
Circle -13840069 true false 60 90 60
Circle -13840069 true false 90 135 60
Circle -13840069 true false 150 135 60
Circle -13840069 true false 120 180 60

wheat
false
0
Circle -1 true false 0 0 300
Rectangle -1184463 true false 135 90 165 300
Polygon -6459832 true false 135 255 90 210 45 195 75 255 135 285
Polygon -6459832 true false 165 255 210 210 255 195 225 255 165 285
Polygon -6459832 true false 135 180 90 135 45 120 75 180 135 210
Polygon -6459832 true false 165 180 165 210 225 180 255 120 210 135
Polygon -6459832 true false 135 105 90 60 45 45 75 105 135 135
Polygon -6459832 true false 165 105 165 135 225 105 255 45 210 60
Polygon -6459832 true false 135 90 120 45 150 15 180 45 165 90

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="AGENTS v140524 -  Yield and Area Eff vs ROR" repetitions="40" runMetricsEveryStep="false">
    <preExperiment>reset-timer</preExperiment>
    <setup>setup</setup>
    <go>go</go>
    <postExperiment>show timer</postExperiment>
    <metric>current-year-index + 1952</metric>
    <metric>yearly-total-precipitation</metric>
    <metric>count terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>count terraces with [crop-type = "grapevine"]</metric>
    <metric>count terraces with [crop-type = "wheat"]</metric>
    <metric>mean [last annual-terrace-yield-efficiency] of terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>mean [wheat-storage] of farms</metric>
    <runMetricsCondition>(ticks &gt;= 1 and day = 31 and current-month = 12) or (ticks = ((model-run-period * 365) - 1))</runMetricsCondition>
    <enumeratedValueSet variable="number-of-farms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="model-run-period">
      <value value="10"/>
    </enumeratedValueSet>
    <steppedValueSet variable="runoff-ratio" first="5" step="1" last="30"/>
    <enumeratedValueSet variable="consecutive-droughts">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farms-memory-length">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-conditions">
      <value value="&quot;same as present-day (87mm)&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farm-wheat-reserve">
      <value value="0.1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="AGENTS 030524 -  efficiency vs farms v1" repetitions="1" runMetricsEveryStep="false">
    <preExperiment>reset-timer</preExperiment>
    <setup>setup</setup>
    <go>go</go>
    <postExperiment>show timer</postExperiment>
    <metric>current-year-index + 1952</metric>
    <metric>yearly-total-precipitation</metric>
    <metric>count terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>mean [last annual-terrace-yield-efficiency] of terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>mean [wheat-storage] of farms</metric>
    <metric>count terraces with [crop-type = "grapevine"]</metric>
    <metric>count terraces with [crop-type = "wheat"]</metric>
    <metric>count patches with [coverage-status = "clear with mounds"]</metric>
    <metric>agrisys-annual-wine-production</metric>
    <runMetricsCondition>(ticks &gt;= 1 and day = 31 and current-month = 12) or (ticks = ((model-run-period * 365) - 1))</runMetricsCondition>
    <steppedValueSet variable="number-of-farms" first="30" step="5" last="50"/>
    <enumeratedValueSet variable="model-run-period">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="runoff-ratio">
      <value value="16"/>
      <value value="18"/>
      <value value="21"/>
      <value value="22"/>
      <value value="25"/>
      <value value="26"/>
      <value value="27"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consecutive-droughts">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farms-memory-length">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-conditions">
      <value value="&quot;same as present-day (87mm)&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farm-wheat-reserve">
      <value value="0.1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="AGENTS 130524 - wine production vs consecutive droughts" repetitions="10" runMetricsEveryStep="false">
    <preExperiment>reset-timer</preExperiment>
    <setup>setup</setup>
    <go>go</go>
    <postExperiment>show timer</postExperiment>
    <metric>current-year-index + 1952</metric>
    <metric>yearly-total-precipitation</metric>
    <metric>count terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>mean [last annual-terrace-yield-efficiency] of terraces with [crop-type = "grapevine" and is-active]</metric>
    <metric>mean [wheat-storage] of farms</metric>
    <metric>count terraces with [crop-type = "grapevine"]</metric>
    <metric>count terraces with [crop-type = "wheat"]</metric>
    <metric>count patches with [coverage-status = "clear with mounds"]</metric>
    <metric>agrisys-annual-wine-production</metric>
    <runMetricsCondition>(ticks &gt;= 1 and day = 31 and current-month = 12) or (ticks = ((model-run-period * 365) - 1))</runMetricsCondition>
    <enumeratedValueSet variable="number-of-farms">
      <value value="17"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="model-run-period">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="runoff-ratio">
      <value value="27"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consecutive-droughts">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drought-threshold">
      <value value="66"/>
    </enumeratedValueSet>
    <steppedValueSet variable="consecutive-droughts-period" first="2" step="1" last="5"/>
    <enumeratedValueSet variable="farms-memory-length">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="climate-conditions">
      <value value="&quot;same as present-day (87mm)&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="farm-wheat-reserve">
      <value value="0.1"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
